<?php
// cash_boxes.php
$page_title = "إدارة الصناديق النقدية";
require_once 'config.php';

// بدء الجلسة إذا لم تكن بدأت
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// عرض رسائل الجلسة المؤقتة
if(isset($_SESSION['success_message'])) {
    $message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
} else {
    $message = '';
}

$action = $_POST['action'] ?? '';
$edit_id = $_GET['edit'] ?? 0;
$delete_id = $_GET['delete'] ?? 0;

// معالجة إنشاء صندوق نقدي
if($action === 'create_cash_box'){
    $name = sanitize($_POST['name']);
    $description = sanitize($_POST['description'] ?? '');
    $opening = (float)$_POST['opening_balance'];
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    // التحقق من عدم تكرار اسم الصندوق
    $check_sql = "SELECT id FROM cash_boxes WHERE name = ?";
    $check_stmt = $mysqli->prepare($check_sql);
    $check_stmt->bind_param('s', $name);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if($check_result->num_rows > 0) {
        $message = "اسم الصندوق هذا مسجل مسبقاً!";
    } else {
        $sql = "INSERT INTO cash_boxes (name, description, opening_balance, current_balance, is_active, created_at) VALUES (?, ?, ?, ?, ?, ?)";
        $stmt = $mysqli->prepare($sql);
        $t = date('Y-m-d H:i:s');
        $stmt->bind_param('ssddis', $name, $description, $opening, $opening, $is_active, $t);
        
        if($stmt->execute()){
            $cash_id = $stmt->insert_id;
            
            if($opening != 0){
                $desc = "رصيد افتتاحي للصندوق: $name";
                $newbal = insert_transaction($mysqli, date('Y-m-d'), 'cash', 'receipt', $cash_id, null, null, $opening, 0.00, $desc, 'cash');
                update_container_balance($mysqli, 'cash_boxes', $cash_id, $newbal);
            }
            
            log_activity($mysqli, current_user_id(), 'create', 'cash_box', $cash_id, null, json_encode(['name'=>$name,'opening_balance'=>$opening]));
            
            $_SESSION['success_message'] = 'تم إنشاء الصندوق النقدي بنجاح.';
            header("Location: cash_boxes.php");
            exit();
        } else {
            $message = 'حدث خطأ أثناء حفظ الصندوق: ' . $stmt->error;
        }
        $stmt->close();
    }
    $check_stmt->close();
}

// معالجة تحديث صندوق نقدي
if($action === 'update_cash_box'){
    $id = (int)$_POST['id'];
    $name = sanitize($_POST['name']);
    $description = sanitize($_POST['description'] ?? '');
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    // التحقق من عدم تكرار اسم الصندوق (باستثناء الصندوق الحالي)
    $check_sql = "SELECT id FROM cash_boxes WHERE name = ? AND id != ?";
    $check_stmt = $mysqli->prepare($check_sql);
    $check_stmt->bind_param('si', $name, $id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if($check_result->num_rows > 0) {
        $message = "اسم الصندوق هذا مسجل مسبقاً لصندوق آخر!";
    } else {
        $res = $mysqli->query("SELECT name, description, opening_balance, current_balance, is_active FROM cash_boxes WHERE id=$id LIMIT 1");
        $old = $res->fetch_assoc();
        
        $sql = "UPDATE cash_boxes SET name=?, description=?, is_active=?, updated_at=? WHERE id=?";
        $stmt = $mysqli->prepare($sql);
        $t = date('Y-m-d H:i:s');
        $stmt->bind_param('ssisi', $name, $description, $is_active, $t, $id);
        
        if($stmt->execute()){
            log_activity($mysqli, current_user_id(), 'update', 'cash_box', $id, json_encode($old), json_encode(['name'=>$name,'is_active'=>$is_active]));
            
            $_SESSION['success_message'] = 'تم تحديث بيانات الصندوق النقدي بنجاح.';
            header("Location: cash_boxes.php");
            exit();
        } else {
            $message = 'حدث خطأ أثناء تحديث الصندوق: ' . $stmt->error;
        }
        $stmt->close();
    }
    $check_stmt->close();
}

// معالجة حذف الصندوق
if($delete_id > 0) {
    // التحقق من عدم وجود معاملات مرتبطة بالصندوق
    $check_transactions = $mysqli->query("SELECT COUNT(*) as count FROM transactions WHERE container_type = 'cash' AND container_id = $delete_id");
    $result = $check_transactions->fetch_assoc();
    
    if($result['count'] > 0) {
        $message = "لا يمكن حذف الصندوق لأنه مرتبط بمعاملات مالية!";
    } else {
        $mysqli->query("DELETE FROM cash_boxes WHERE id = $delete_id");
        log_activity($mysqli, current_user_id(), 'delete', 'cash_box', $delete_id, null, null);
        
        $_SESSION['success_message'] = 'تم حذف الصندوق النقدي بنجاح.';
        header("Location: cash_boxes.php");
        exit();
    }
}

// جلب بيانات الصندوق للتعديل
$editing_box = null;
if($edit_id > 0) {
    $result = $mysqli->query("SELECT * FROM cash_boxes WHERE id = $edit_id");
    if($result->num_rows > 0) {
        $editing_box = $result->fetch_assoc();
    }
}

require_once 'header.php';
?>

<div class="card" style="border: none; box-shadow: 0 2px 10px rgba(0,0,0,0.08); border-radius: 0;padding: 8px;  margin: 5px auto; max-width: 1200px;">
 <div class="card-header" style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f; margin-bottom: 0;  position: sticky; top: 0;
z-index: 1100;">
        <div style="display: flex; justify-content: space-between; align-items: center;">
            <h2 style="margin: 0; font-weight: 600; font-size: 18px;">
                <i class="fas fa-cash-register" style="margin-left: 8px;"></i>
                إدارة الصناديق النقدية
            </h2>
            <button type="button" class="btn" onclick="showAddForm()" 
                   style="padding: 8px 16px; background: #2ecc71; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center; transition: background 0.3s;"
                   onmouseover="this.style.background='#27ae60'"
                   onmouseout="this.style.background='#2ecc71'">
                <i class="fas fa-plus" style="margin-left: 8px;"></i>
                إضافة صندوق نقدي جديد
            </button>
        </div>
    </div>

    <div class="card-body" style="padding: 0;">
        <!-- شريط البحث والإحصائيات -->
        <div style="padding: 20px; border-bottom: 1px solid #ecf0f1;">
            <!-- شريط البحث -->
            <form method="GET" action="" style="margin-bottom: 10px;">
                <div style="display: flex; gap: 10px;">
                    <div style="position: relative; flex: 1;">
                        <input type="text" name="search" class="form-control" 
                               placeholder="ابحث باسم الصندوق..." 
                               value="<?php echo htmlspecialchars($search ?? ''); ?>"
                               style="width: 100%; padding: 10px 12px 10px 40px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px;">
                        <i class="fas fa-search" style="position: absolute; left: 12px; top: 50%; transform: translateY(-50%); color: #7f8c8d;"></i>
                    </div>
                    <button type="submit" class="btn" 
                            style="padding: 8px 20px; background: #3498db; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; display: inline-flex; align-items: center;">
                        <i class="fas fa-search" style="margin-left: 8px;"></i>
                        بحث
                    </button>
                    <?php if (!empty($search)): ?>
                        <a href="cash_boxes.php" class="btn" 
                           style="padding: 8px 16px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center;">
                            <i class="fas fa-times" style="margin-left: 8px;"></i>
                            إلغاء البحث
                        </a>
                    <?php endif; ?>
                </div>
            </form>

            <!-- الإحصائيات -->
            <?php
            // حساب الإحصائيات
            $result = $mysqli->query("SELECT 
                COUNT(*) as total_boxes,
                SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_boxes,
                SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive_boxes,
                SUM(current_balance) as total_balance
                FROM cash_boxes");
            $stats = $result->fetch_assoc();
            ?>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px;">
                <div style="background: #e8f6f3; padding: 15px; border-left: 4px solid #1abc9c;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">إجمالي الصناديق</div>
                    <div style="font-size: 24px; font-weight: bold; color: #16a085;"><?php echo number_format($stats['total_boxes'] ?? 0); ?></div>
                </div>
                <div style="background: #fef9e7; padding: 15px; border-left: 4px solid #f1c40f;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">صناديق مفعلة</div>
                    <div style="font-size: 24px; font-weight: bold; color: #f39c12;"><?php echo number_format($stats['active_boxes'] ?? 0); ?></div>
                </div>
                <div style="background: #fdedec; padding: 15px; border-left: 4px solid #e74c3c;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">صناديق غير مفعلة</div>
                    <div style="font-size: 24px; font-weight: bold; color: #c0392b;"><?php echo number_format($stats['inactive_boxes'] ?? 0); ?></div>
                </div>
                <div style="background: #e8f4fd; padding: 15px; border-left: 4px solid #3498db;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">إجمالي الأرصدة</div>
                    <div style="font-size: 24px; font-weight: bold; color: #2980b9;">
                        <?php echo number_format($stats['total_balance'] ?? 0, 2); ?> 
                        <span style="font-size: 14px;">¥</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- رسائل النجاح والخطأ -->
        <?php if ($message): ?>
            <div style="margin: 10px 12px; padding: 8px 12px; background: #d4edda; color: #155724; border: 1px solid #c3e6cb; border-radius: 0; display: flex; align-items: center; justify-content: space-between;">
                <div style="display: flex; align-items: center;">
                    <i class="fas fa-check-circle" style="margin-left: 8px; color: #28a745;"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
                <button onclick="this.parentElement.style.display='none'" style="background: none; border: none; color: #155724; cursor: pointer;">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        <?php endif; ?>

        <!-- حاوية أزرار الإجراءات (ستظهر عند التحديد) -->
        <div id="actionsContainer" style="max-height: 0; overflow: hidden; transition: max-height 0.3s ease; margin: 0 20px;">
            <div style="display: flex; gap: 8px; padding: 15px; background: #f8f9fa; border-bottom: 1px solid #ecf0f1; flex-wrap: wrap;">
                <a href="javascript:void(0)" class="btn" id="editBtn"
                   style="padding: 6px 12px; background: #3498db; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center; transition: background 0.3s;"
                   onmouseover="this.style.background='#2980b9'"
                   onmouseout="this.style.background='#3498db'">
                    <i class="fas fa-edit" style="margin-left: 5px; font-size: 11px;"></i>
                    تعديل
                </a>

                <button type="button" class="btn" id="viewBtn"
                        style="padding: 6px 12px; background: #17a2b8; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; display: inline-flex; align-items: center; transition: background 0.3s;"
                        onmouseover="this.style.background='#138496'"
                        onmouseout="this.style.background='#17a2b8'">
                    <i class="fas fa-eye" style="margin-left: 5px; font-size: 11px;"></i>
                    عرض التفاصيل
                </button>

                <button type="button" class="btn" id="transactionsBtn"
                        style="padding: 6px 12px; background: #9b59b6; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; display: inline-flex; align-items: center; transition: background 0.3s;"
                        onmouseover="this.style.background='#8e44ad'"
                        onmouseout="this.style.background='#9b59b6'">
                    <i class="fas fa-exchange-alt" style="margin-left: 5px; font-size: 11px;"></i>
                    المعاملات
                </button>

                <button type="button" class="btn" id="toggleStatusBtn"
                        style="padding: 6px 12px; background: #28a745; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; display: inline-flex; align-items: center; transition: background 0.3s;"
                        onmouseover="this.style.background='#218838'"
                        onmouseout="this.style.background='#28a745'">
                    <i class="fas fa-power-off" style="margin-left: 5px; font-size: 11px;"></i>
                    <span id="toggleStatusText">تفعيل</span>
                </button>

                <button type="button" class="btn" id="deleteBtn"
                        style="padding: 6px 12px; background: #e74c3c; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; display: inline-flex; align-items: center; transition: background 0.3s;"
                        onmouseover="this.style.background='#c0392b'"
                        onmouseout="this.style.background='#e74c3c'">
                    <i class="fas fa-trash" style="margin-left: 5px; font-size: 11px;"></i>
                    حذف
                </button>
            </div>
        </div>

        <!-- جدول الصناديق النقدية -->
        <div style="overflow-x: auto;">
            <table class="table" style="width: 100%; border-collapse: collapse;">
                <thead>
                    <tr style="background: #f8f9fa;">
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">#</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: right; font-weight: 600; color: #F19E2C; font-size: 14px;">اسم الصندوق</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: right; font-weight: 600; color: #F19E2C; font-size: 14px;">الوصف</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">الرصيد الحالي</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">الرصيد الافتتاحي</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">الحالة</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">تاريخ الإنشاء</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $cash_boxes = get_cash_boxes($mysqli);
                    $total_balance = 0;
                    $counter = 0;
                    
                    if($cash_boxes->num_rows > 0): 
                        while($cb = $cash_boxes->fetch_assoc()): 
                            $counter++;
                            $total_balance += $cb['current_balance'];
                    ?>
                    <tr style="border-bottom: 1px solid #ecf0f1; transition: background 0.3s; cursor: pointer;" 
                        onmouseover="this.style.backgroundColor='#f8f9fa'" 
                        onmouseout="this.style.backgroundColor='white'"
                        onclick="selectBox(<?php echo $cb['id']; ?>, this)"
                        data-box-id="<?php echo $cb['id']; ?>"
                        data-box-name="<?php echo htmlspecialchars($cb['name']); ?>"
                        data-box-description="<?php echo htmlspecialchars($cb['description']); ?>"
                        data-box-status="<?php echo $cb['is_active']; ?>"
                        data-box-balance="<?php echo $cb['current_balance']; ?>"
                        data-box-opening="<?php echo $cb['opening_balance']; ?>"
                        data-box-created="<?php echo $cb['created_at']; ?>">
                        
                        <td style="padding: 12px; text-align: center; color: #7f8c8d; font-size: 14px; vertical-align: middle;">
                            <?php echo $counter; ?>
                        </td>
                        
                        <td style="padding: 12px; vertical-align: middle;">
                            <div style="display: flex; align-items: center; justify-content: flex-end;">
                                <div style="text-align: right;">
                                    <strong style="color: #2c3e50; font-size: 14px;">
                                        <?php echo htmlspecialchars($cb['name']); ?>
                                    </strong>
                                </div>
                            </div>
                        </td>
                        
                        <td style="padding: 12px; text-align: right; vertical-align: middle;">
                            <div style="color: #7f8c8d; font-size: 13px; max-width: 250px; overflow: hidden; text-overflow: ellipsis;">
                                <?php echo !empty($cb['description']) ? htmlspecialchars($cb['description']) : '<span style="color: #bdc3c7;">-</span>'; ?>
                            </div>
                        </td>
                        
                        <td style="padding: 12px; text-align: center; vertical-align: middle;">
                            <div style="font-weight: 600; color: #27ae60; font-size: 14px;">
                                <?php echo number_format($cb['current_balance'], 2); ?> 
                                <span style="font-size: 12px; color: #7f8c8d;">¥</span>
                            </div>
                        </td>
                        
                        <td style="padding: 12px; text-align: center; vertical-align: middle;">
                            <div style="color: #3498db; font-size: 13px;">
                                <?php echo number_format($cb['opening_balance'], 2); ?> 
                                <span style="font-size: 11px; color: #95a5a6;">¥</span>
                            </div>
                        </td>
                        
                        <td style="padding: 12px; text-align: center; vertical-align: middle;">
                            <?php if($cb['is_active']): ?>
                                <span style="background: #2ecc71; color: white; padding: 4px 10px; border-radius: 2px; font-size: 12px; font-weight: 600;">
                                    <i class="fas fa-check-circle" style="margin-left: 5px;"></i>
                                    مفعل
                                </span>
                            <?php else: ?>
                                <span style="background: #e74c3c; color: white; padding: 4px 10px; border-radius: 2px; font-size: 12px; font-weight: 600;">
                                    <i class="fas fa-times-circle" style="margin-left: 5px;"></i>
                                    غير مفعل
                                </span>
                            <?php endif; ?>
                        </td>
                        
                        <td style="padding: 12px; text-align: center; vertical-align: middle;">
                            <div style="color: #34495e; font-size: 13px;">
                                <?php echo date('Y-m-d', strtotime($cb['created_at'])); ?>
                            </div>
                            <div style="font-size: 11px; color: #95a5a6;">
                                <?php echo date('H:i', strtotime($cb['created_at'])); ?>
                            </div>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="7" style="padding: 20px; text-align: center; color: #7f8c8d; font-size: 14px;">
                                <i class="fas fa-cash-register" style="font-size: 48px; color: #ecf0f1; margin-bottom: 15px; display: block;"></i>
                                لا توجد صناديق نقدية
                                <?php if (!empty($search)): ?>
                                    <div style="margin-top: 10px; font-size: 13px;">
                                        لم يتم العثور على نتائج لبحثك: "<?php echo htmlspecialchars($search ?? ''); ?>"
                                    </div>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <!-- الترويج والتذييل -->
        <div style="padding: 20px; border-top: 1px solid #ecf0f1; background: #f8f9fa;">
            <?php if ($counter > 0): ?>
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div style="font-size: 13px; color: #7f8c8d;">
                    <i class="fas fa-info-circle" style="margin-left: 5px;"></i>
                    إجمالي الصناديق: <?php echo number_format($counter); ?> صندوق
                </div>
                <div style="font-size: 13px; color: #7f8c8d;">
                    <i class="fas fa-calculator" style="margin-left: 5px;"></i>
                    إجمالي الأرصدة: <?php echo number_format($total_balance, 2); ?> ¥
                </div>
            </div>
            <?php else: ?>
                <div style="text-align: center; color: #7f8c8d; font-size: 13px;">
                    <i class="fas fa-info-circle" style="margin-left: 5px;"></i>
                    لا توجد صناديق نقدية لعرضها
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- نافذة إضافة/تعديل صندوق -->
<div id="boxFormModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
    <div style="background: white; width: 90%; max-width: 500px; max-height: 90vh; overflow-y: auto; border-radius: 0; box-shadow: 0 4px 20px rgba(0,0,0,0.2);">
        <div style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                    <i class="fas fa-cash-register" style="margin-left: 8px;"></i>
                    <span id="formModalTitle">إضافة صندوق نقدي جديد</span>
                </h3>
                <button onclick="hideBoxFormModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
            </div>
        </div>
        <div style="padding: 20px;">
            <form method="post" id="cashForm" onsubmit="return validateCashForm()">
                <input type="hidden" name="action" id="formAction" value="create_cash_box" />
                <input type="hidden" name="id" id="formBoxId" value="" />
                
                <div style="margin-bottom: 15px;">
                    <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                        <span style="color: #e74c3c;">*</span> اسم الصندوق
                    </label>
                    <input type="text" name="name" id="formName" class="form-control" 
                           style="width: 100%; padding: 10px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px;"
                           required placeholder="مثال: الصندوق الرئيسي" />
                    <small style="font-size: 11px; color: #7f8c8d;">سيتم التحقق من عدم تكرار اسم الصندوق</small>
                </div>
                
                <div style="margin-bottom: 15px;">
                    <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                        <span style="color: #e74c3c;">*</span> الرصيد الافتتاحي
                    </label>
                    <div style="position: relative;">
                        <input type="number" name="opening_balance" id="formOpeningBalance" class="form-control"
                               style="width: 100%; padding: 10px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px; padding-left: 40px;"
                               step="0.01" required placeholder="0.00" value="0" />
                        <span style="position: absolute; left: 12px; top: 50%; transform: translateY(-50%); color: #7f8c8d;">¥</span>
                    </div>
                </div>
                
                <div style="margin-bottom: 15px;">
                    <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                        وصف الصندوق (اختياري)
                    </label>
                    <textarea name="description" id="formDescription" class="form-control"
                              style="width: 100%; padding: 10px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px; resize: vertical; min-height: 80px;"
                              placeholder="أدخل وصفاً للصندوق..."></textarea>
                </div>
                
                <div style="display: flex; align-items: center; margin-bottom: 20px;">
                    <label style="display: flex; align-items: center; gap: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                        <input type="checkbox" name="is_active" id="formIsActive" value="1" checked 
                               style="width: 18px; height: 18px;" />
                        الصندوق مفعل
                    </label>
                </div>
                
                <div style="display: flex; gap: 10px; justify-content: flex-end; margin-top: 25px; padding-top: 20px; border-top: 1px solid #ecf0f1;">
                    <button type="button" onclick="hideBoxFormModal()" 
                            style="padding: 8px 20px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;">
                        إلغاء
                    </button>
                    <button type="submit" 
                            style="padding: 8px 20px; background: #2ecc71; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; display: inline-flex; align-items: center;">
                        <i class="fas fa-save" style="margin-left: 8px;"></i>
                        <span id="formSubmitText">حفظ الصندوق</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- نافذة تأكيد الحذف -->
<div id="deleteModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
    <div style="background: white; width: 90%; max-width: 400px; border-radius: 0; box-shadow: 0 4px 20px rgba(0,0,0,0.2);">
        <div style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                    <i class="fas fa-trash" style="margin-left: 8px;"></i>
                    تأكيد الحذف
                </h3>
                <button onclick="hideDeleteModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
            </div>
        </div>
        <div style="padding: 20px;">
            <div style="text-align: center; margin-bottom: 20px;">
                <i class="fas fa-exclamation-triangle" style="font-size: 48px; color: #e74c3c; margin-bottom: 15px;"></i>
                <p style="font-size: 16px; color: #2c3e50; margin-bottom: 10px;">هل أنت متأكد من حذف الصندوق النقدي؟</p>
                <p style="font-size: 14px; color: #7f8c8d;" id="deleteMessage">
                    سيتم حذف الصندوق النقدي <strong id="boxName"></strong> ولا يمكن التراجع عن هذا الإجراء.
                </p>
                <p style="font-size: 13px; color: #e74c3c; margin-top: 10px; padding: 8px; background: #fdedec; border-radius: 0;">
                    <i class="fas fa-exclamation-circle" style="margin-left: 5px;"></i>
                    لا يمكن حذف الصندوق إذا كان مرتبطاً بمعاملات مالية.
                </p>
            </div>
            <div style="display: flex; gap: 10px; justify-content: flex-end;">
                <button onclick="hideDeleteModal()" style="padding: 8px 20px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;">
                    إلغاء
                </button>
                <a href="#" class="btn" id="deleteLink"
                   style="padding: 8px 20px; background: #e74c3c; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center;">
                    نعم، احذف
                </a>
            </div>
        </div>
    </div>
</div>

<!-- نافذة عرض تفاصيل الصندوق -->
<div id="viewBoxModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
    <div style="background: white; width: 90%; max-width: 500px; max-height: 90vh; border-radius: 0; box-shadow: 0 4px 20px rgba(0,0,0,0.2); display: flex; flex-direction: column;">
        <div style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                    <i class="fas fa-cash-register" style="margin-left: 8px;"></i>
                    تفاصيل الصندوق النقدي
                </h3>
                <button onclick="hideViewBoxModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
            </div>
        </div>
        <div style="padding: 20px; overflow-y: auto; flex: 1;" id="boxDetails">
            <!-- سيتم تعبئة البيانات هنا عبر JavaScript -->
        </div>
        <div style="padding: 15px 20px; border-top: 1px solid #ecf0f1; background: #f8f9fa; display: flex; justify-content: flex-end; gap: 10px;">
            <button onclick="hideViewBoxModal()" style="padding: 8px 20px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;">
                إغلاق
            </button>
        </div>
    </div>
</div>

<!-- نافذة تأكيد تغيير الحالة -->
<div id="statusModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
    <div style="background: white; width: 90%; max-width: 400px; border-radius: 0; box-shadow: 0 4px 20px rgba(0,0,0,0.2);">
        <div style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                    <i class="fas fa-power-off" style="margin-left: 8px;"></i>
                    تغيير حالة الصندوق
                </h3>
                <button onclick="hideStatusModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
            </div>
        </div>
        <div style="padding: 20px;">
            <div style="text-align: center; margin-bottom: 20px;">
                <i class="fas fa-question-circle" style="font-size: 48px; color: #3498db; margin-bottom: 15px;"></i>
                <p style="font-size: 16px; color: #2c3e50; margin-bottom: 10px;" id="statusModalTitle">تأكيد تفعيل الصندوق</p>
                <p style="font-size: 14px; color: #7f8c8d;" id="statusModalMessage">
                    هل أنت متأكد من تغيير حالة الصندوق <strong id="statusBoxName"></strong>؟
                </p>
            </div>
            <div style="display: flex; gap: 10px; justify-content: flex-end;">
                <button onclick="hideStatusModal()" style="padding: 8px 20px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;">
                    إلغاء
                </button>
                <form method="POST" id="statusForm" style="margin: 0;">
                    <input type="hidden" name="action" value="update_cash_box_status">
                    <input type="hidden" name="box_id" id="statusBoxId">
                    <input type="hidden" name="new_status" id="newStatus">
                    <button type="submit" style="padding: 8px 20px; background: #3498db; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;" id="statusConfirmBtn">
                        نعم، تأكيد
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<style>
    /* تنسيقات إضافية متناسقة مع واجهة الفواتير */
    .form-control:focus {
        outline: none;
        border-color: #3498db;
        box-shadow: 0 0 0 2px rgba(52,152,219,0.2);
    }
    
    .table tbody tr {
        transition: background-color 0.2s ease;
    }
    
    .table tbody tr.selected {
        background-color: rgba(14, 66, 98, 0.1) !important;
        border-left: 3px solid #F19E2C;
    }
    
    /* تخصيص شريط التمرير */
    ::-webkit-scrollbar {
        width: 8px;
        height: 8px;
    }
    
    ::-webkit-scrollbar-track {
        background: #f1f1f1;
    }
    
    ::-webkit-scrollbar-thumb {
        background: #bdc3c7;
    }
    
    ::-webkit-scrollbar-thumb:hover {
        background: #95a5a6;
    }
    
    /* إخفاء بعض الأعمدة في الجوال */
    @media (max-width: 768px) {
        .table thead th:nth-child(3), /* الوصف */
        .table tbody td:nth-child(3),
        .table thead th:nth-child(5), /* الرصيد الافتتاحي */
        .table tbody td:nth-child(5),
        .table thead th:nth-child(7), /* تاريخ الإنشاء */
        .table tbody td:nth-child(7) {
            display: none;
        }
        
        .table thead th,
        .table tbody td {
            padding: 8px 5px !important;
        }
        
        #boxFormModal > div {
            width: 95% !important;
        }
        
        #boxFormModal > div > div:last-child {
            padding: 15px !important;
        }
    }
    
    /* تنسيق textarea */
    textarea.form-control {
        resize: vertical;
        min-height: 80px;
        max-height: 150px;
    }
</style>

<script>
// متغيرات عالمية لتخزين الصندوق المحدد
let selectedBoxId = null;
let selectedBoxName = null;
let selectedBoxElement = null;
let selectedBoxStatus = null;

// اختيار صندوق
function selectBox(boxId, element) {
    // إلغاء تحديد الصف السابق إذا كان موجوداً
    if (selectedBoxElement) {
        selectedBoxElement.classList.remove('selected');
    }

    // تحديد الصف الجديد
    element.classList.add('selected');
    selectedBoxElement = element;

    // حفظ بيانات الصندوق المحدد
    selectedBoxId = boxId;
    selectedBoxName = element.getAttribute('data-box-name');
    selectedBoxStatus = element.getAttribute('data-box-status');

    // تحديث روابط الأزرار
    updateActionButtons();

    // إظهار حاوية الأزرار
    showActionsContainer();

    // تمرير سلس إلى الصف المحدد في الجوال
    if (window.innerWidth <= 768) {
        setTimeout(() => {
            const actionsContainer = document.getElementById('actionsContainer');
            actionsContainer.style.maxHeight = actionsContainer.scrollHeight + 'px';
            actionsContainer.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }, 300);
    }
}

// تحديث روابط الأزرار بناءً على الصندوق المحدد
function updateActionButtons() {
    if (!selectedBoxId) return;

    // تحديث روابط الأزرار
    document.getElementById('editBtn').onclick = () => editBox(selectedBoxId);
    document.getElementById('viewBtn').onclick = () => viewBox(selectedBoxId);
    document.getElementById('toggleStatusBtn').onclick = () => toggleBoxStatus(selectedBoxId, selectedBoxStatus);
    document.getElementById('deleteBtn').onclick = () => confirmDelete(selectedBoxId, selectedBoxName);

    // تحديث زر تغيير الحالة
    updateStatusButton();
}

// تحديث زر تغيير الحالة
function updateStatusButton() {
    const statusBtn = document.getElementById('toggleStatusBtn');
    const statusBtnText = document.getElementById('toggleStatusText');
    
    if (!selectedBoxStatus) return;

    if (selectedBoxStatus === '0') {
        statusBtn.style.backgroundColor = '#28a745';
        statusBtn.onmouseover = function() { this.style.backgroundColor = '#218838'; };
        statusBtn.onmouseout = function() { this.style.backgroundColor = '#28a745'; };
        statusBtnText.textContent = 'تفعيل';
    } else {
        statusBtn.style.backgroundColor = '#f39c12';
        statusBtn.onmouseover = function() { this.style.backgroundColor = '#e67e22'; };
        statusBtn.onmouseout = function() { this.style.backgroundColor = '#f39c12'; };
        statusBtnText.textContent = 'تعطيل';
    }
}

// إظهار حاوية أزرار الإجراءات
function showActionsContainer() {
    const container = document.getElementById('actionsContainer');
    container.style.maxHeight = container.scrollHeight + 'px';
}

// إخفاء حاوية أزرار الإجراءات
function hideActionsContainer() {
    const container = document.getElementById('actionsContainer');
    container.style.maxHeight = '0';
}

// إظهار نموذج إضافة/تعديل الصندوق
function showAddForm() {
    document.getElementById('formModalTitle').textContent = 'إضافة صندوق نقدي جديد';
    document.getElementById('formAction').value = 'create_cash_box';
    document.getElementById('formSubmitText').textContent = 'حفظ الصندوق';
    document.getElementById('formBoxId').value = '';
    document.getElementById('formName').value = '';
    document.getElementById('formOpeningBalance').value = '0';
    document.getElementById('formDescription').value = '';
    document.getElementById('formIsActive').checked = true;
    
    document.getElementById('boxFormModal').style.display = 'flex';
}

// تعديل صندوق
function editBox(boxId) {
    const row = document.querySelector(`tr[data-box-id="${boxId}"]`);
    if (!row) return;

    document.getElementById('formModalTitle').textContent = 'تعديل صندوق نقدي';
    document.getElementById('formAction').value = 'update_cash_box';
    document.getElementById('formSubmitText').textContent = 'تحديث الصندوق';
    document.getElementById('formBoxId').value = boxId;
    document.getElementById('formName').value = row.getAttribute('data-box-name');
    document.getElementById('formOpeningBalance').value = row.getAttribute('data-box-opening');
    document.getElementById('formDescription').value = row.getAttribute('data-box-description');
    document.getElementById('formIsActive').checked = row.getAttribute('data-box-status') === '1';
    
    document.getElementById('boxFormModal').style.display = 'flex';
}

// عرض تفاصيل الصندوق
function viewBox(boxId) {
    const row = document.querySelector(`tr[data-box-id="${boxId}"]`);
    if (!row) return;

    const status = row.getAttribute('data-box-status') === '1' ? 
        '<span style="background: #2ecc71; color: white; padding: 4px 10px; border-radius: 2px; font-size: 12px; font-weight: 600;">مفعل</span>' : 
        '<span style="background: #e74c3c; color: white; padding: 4px 10px; border-radius: 2px; font-size: 12px; font-weight: 600;">غير مفعل</span>';

    const description = row.getAttribute('data-box-description') ? 
        `<div style="color: #7f8c8d; font-size: 14px; line-height: 1.6;">${row.getAttribute('data-box-description')}</div>` : 
        '<div style="color: #bdc3c7; font-size: 14px; font-style: italic;">لا يوجد وصف</div>';

    const createdDate = new Date(row.getAttribute('data-box-created'));
    const formattedDate = createdDate.toLocaleDateString('ar-EG');
    const formattedTime = createdDate.toLocaleTimeString('ar-EG', { hour: '2-digit', minute: '2-digit' });

    document.getElementById('boxDetails').innerHTML = `
        <div style="margin-bottom: 20px;">
            <h4 style="color: #2c3e50; border-bottom: 2px solid #F19E2C; padding-bottom: 8px; margin-bottom: 15px;">
                <i class="fas fa-cash-register" style="margin-left: 8px;"></i>
                معلومات الصندوق النقدي
            </h4>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                <div style="background: #f8f9fa; padding: 15px; border-left: 3px solid #3498db;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">اسم الصندوق</div>
                    <div style="font-size: 16px; font-weight: 600; color: #2c3e50;">${row.getAttribute('data-box-name')}</div>
                </div>
                
                <div style="background: #f8f9fa; padding: 15px; border-left: 3px solid #f39c12;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">الحالة</div>
                    <div style="font-size: 16px; font-weight: 600;">${status}</div>
                </div>
            </div>
        </div>
        
        <div style="margin-bottom: 20px;">
            <h4 style="color: #2c3e50; border-bottom: 2px solid #F19E2C; padding-bottom: 8px; margin-bottom: 15px;">
                <i class="fas fa-money-bill-wave" style="margin-left: 8px;"></i>
                المعلومات المالية
            </h4>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                <div style="background: #e8f6f3; padding: 15px; border-left: 3px solid #1abc9c;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">الرصيد الحالي</div>
                    <div style="font-size: 20px; font-weight: 600; color: #27ae60;">
                        ${parseFloat(row.getAttribute('data-box-balance')).toFixed(2)} ¥
                    </div>
                </div>
                
                <div style="background: #fef9e7; padding: 15px; border-left: 3px solid #f1c40f;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">الرصيد الافتتاحي</div>
                    <div style="font-size: 16px; font-weight: 600; color: #f39c12;">
                        ${parseFloat(row.getAttribute('data-box-opening')).toFixed(2)} ¥
                    </div>
                </div>
            </div>
        </div>
        
        <div style="margin-bottom: 20px;">
            <h4 style="color: #2c3e50; border-bottom: 2px solid #F19E2C; padding-bottom: 8px; margin-bottom: 15px;">
                <i class="fas fa-align-left" style="margin-left: 8px;"></i>
                الوصف
            </h4>
            ${description}
        </div>
        
        <div>
            <h4 style="color: #2c3e50; border-bottom: 2px solid #F19E2C; padding-bottom: 8px; margin-bottom: 15px;">
                <i class="fas fa-calendar" style="margin-left: 8px;"></i>
                معلومات الإضافة
            </h4>
            
            <div style="background: #f8f9fa; padding: 15px; border-left: 3px solid #9b59b6;">
                <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">تاريخ الإنشاء</div>
                <div style="font-size: 16px; font-weight: 600; color: #2c3e50;">
                    ${formattedDate}
                    <span style="font-size: 13px; color: #95a5a6; margin-right: 5px;">${formattedTime}</span>
                </div>
            </div>
        </div>
    `;

    document.getElementById('viewBoxModal').style.display = 'flex';
}

// تغيير حالة الصندوق (تفعيل/تعطيل)
function toggleBoxStatus(boxId, currentStatus) {
    const row = document.querySelector(`tr[data-box-id="${boxId}"]`);
    if (!row) return;

    const newStatus = currentStatus === '1' ? '0' : '1';
    const actionText = newStatus === '1' ? 'تفعيل' : 'تعطيل';
    
    document.getElementById('statusBoxId').value = boxId;
    document.getElementById('newStatus').value = newStatus;
    document.getElementById('statusBoxName').textContent = row.getAttribute('data-box-name');
    document.getElementById('statusModalTitle').textContent = 'تأكيد ' + actionText + ' الصندوق';
    document.getElementById('statusModalMessage').innerHTML =
        `هل أنت متأكد من ${actionText} الصندوق <strong>${row.getAttribute('data-box-name')}</strong>؟`;
    document.getElementById('statusConfirmBtn').textContent = 'نعم، ' + actionText;

    document.getElementById('statusModal').style.display = 'flex';
}

// تأكيد حذف الصندوق
function confirmDelete(boxId, boxName) {
    document.getElementById('boxName').textContent = boxName;
    document.getElementById('deleteMessage').innerHTML =
        `سيتم حذف الصندوق النقدي <strong>${boxName}</strong> ولا يمكن التراجع عن هذا الإجراء.`;
    
    document.getElementById('deleteLink').href = `cash_boxes.php?delete=${boxId}`;
    
    document.getElementById('deleteModal').style.display = 'flex';
}

// التحقق من صحة النموذج
function validateCashForm() {
    const boxName = document.getElementById('formName').value.trim();
    const openingBalance = parseFloat(document.getElementById('formOpeningBalance').value) || 0;
    
    // التحقق من اسم الصندوق
    if (boxName.length < 2) {
        alert('اسم الصندوق يجب أن يكون على الأقل حرفين');
        return false;
    }
    
    // التحقق من الرصيد الافتتاحي
    if (openingBalance < 0) {
        alert('الرصيد الافتتاحي لا يمكن أن يكون سالباً');
        return false;
    }
    
    return true;
}

// وظائف إخفاء النوافذ المنبثقة
function hideBoxFormModal() {
    document.getElementById('boxFormModal').style.display = 'none';
}

function hideDeleteModal() {
    document.getElementById('deleteModal').style.display = 'none';
}

function hideViewBoxModal() {
    document.getElementById('viewBoxModal').style.display = 'none';
}

function hideStatusModal() {
    document.getElementById('statusModal').style.display = 'none';
}

// إغلاق النوافذ المنبثقة عند النقر خارجها
window.onclick = function(event) {
    const modals = ['boxFormModal', 'deleteModal', 'viewBoxModal', 'statusModal'];
    modals.forEach(modalId => {
        const modal = document.getElementById(modalId);
        if (event.target == modal) {
            modal.style.display = 'none';
        }
    });
};

// إضافة تأثيرات تفاعلية عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    // إخفاء حاوية الأزرار عند النقر خارج الجدول
    document.addEventListener('click', function(event) {
        const table = document.querySelector('.table');
        const actionsContainer = document.getElementById('actionsContainer');

        // التحقق إذا كان النقر على أي من أزرار الإجراءات أو داخل النافذة المنبثقة
        const isModal = event.target.closest('#boxFormModal') || 
                        event.target.closest('#deleteModal') ||
                        event.target.closest('#viewBoxModal') ||
                        event.target.closest('#statusModal');
        
        const isActionButton = event.target.closest('#actionsContainer') || 
                              event.target.closest('.btn') ||
                              isModal;

        // إذا تم النقر خارج الجدول وخارج حاوية الأزرار وخارج النوافذ المنبثقة
        if (!table.contains(event.target) && !isActionButton) {
            if (selectedBoxElement) {
                selectedBoxElement.classList.remove('selected');
                selectedBoxElement = null;
                selectedBoxId = null;
                selectedBoxName = null;
                selectedBoxStatus = null;
                hideActionsContainer();
            }
        }
    });

    // إضافة وظيفة البحث السريع
    const searchInput = document.querySelector('input[name="search"]');
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('.table tbody tr');
            
            rows.forEach(row => {
                const boxName = row.getAttribute('data-box-name') || '';
                const description = row.getAttribute('data-box-description') || '';
                
                if (boxName.toLowerCase().includes(searchTerm) || 
                    description.toLowerCase().includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
    }
});
</script>