<?php
// models/UserModel.php
require_once 'core/Database.php';

class UserModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function getAll($page = 1, $perPage = 10) {
        $offset = ($page - 1) * $perPage;
        
        $sql = "SELECT u.*, r.name as role_name 
                FROM users u 
                JOIN roles r ON u.role_id = r.id 
                WHERE u.is_active = 1 
                ORDER BY u.created_at DESC 
                LIMIT :limit OFFSET :offset";
        
        $stmt = $this->db->getConnection()->prepare($sql);
        $stmt->bindValue(':limit', (int)$perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    public function getById($id) {
        $sql = "SELECT u.*, r.name as role_name, r.permissions 
                FROM users u 
                JOIN roles r ON u.role_id = r.id 
                WHERE u.id = ? AND u.is_active = 1";
        
        $stmt = $this->db->query($sql, [$id]);
        return $stmt->fetch();
    }
    
    public function create($data) {
        $this->db->beginTransaction();
        
        try {
            // التحقق من عدم تكرار اسم المستخدم أو البريد الإلكتروني
            $checkSql = "SELECT id FROM users WHERE username = ? OR email = ?";
            $checkStmt = $this->db->query($checkSql, [$data['username'], $data['email']]);
            if ($checkStmt->fetch()) {
                throw new Exception("اسم المستخدم أو البريد الإلكتروني موجود مسبقاً");
            }
            
            $sql = "INSERT INTO users (username, email, password, role_id, created_at) 
                    VALUES (?, ?, ?, ?, NOW())";
            
            $passwordHash = password_hash($data['password'], PASSWORD_DEFAULT);
            
            $this->db->query($sql, [
                $data['username'],
                $data['email'],
                $passwordHash,
                $data['role_id']
            ]);
            
            $userId = $this->db->lastInsertId();
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'create', 'user', $userId, null, [
                'username' => $data['username'],
                'email' => $data['email'],
                'role_id' => $data['role_id']
            ]);
            
            $this->db->commit();
            
            return $userId;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function update($id, $data) {
        $this->db->beginTransaction();
        
        try {
            // الحصول على البيانات القديمة
            $oldUser = $this->getById($id);
            
            // التحقق من عدم تكرار اسم المستخدم أو البريد الإلكتروني
            $checkSql = "SELECT id FROM users WHERE (username = ? OR email = ?) AND id != ?";
            $checkStmt = $this->db->query($checkSql, [$data['username'], $data['email'], $id]);
            if ($checkStmt->fetch()) {
                throw new Exception("اسم المستخدم أو البريد الإلكتروني موجود مسبقاً");
            }
            
            $sql = "UPDATE users SET username = ?, email = ?, role_id = ?, updated_at = NOW()";
            $params = [$data['username'], $data['email'], $data['role_id']];
            
            if (!empty($data['password'])) {
                $sql .= ", password = ?";
                $params[] = password_hash($data['password'], PASSWORD_DEFAULT);
            }
            
            $sql .= " WHERE id = ?";
            $params[] = $id;
            
            $this->db->query($sql, $params);
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'update', 'user', $id, $oldUser, [
                'username' => $data['username'],
                'email' => $data['email'],
                'role_id' => $data['role_id']
            ]);
            
            $this->db->commit();
            
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function delete($id) {
        $user = $this->getById($id);
        
        $sql = "UPDATE users SET is_active = 0, updated_at = NOW() WHERE id = ?";
        $result = $this->db->query($sql, [$id]);
        
        // تسجيل النشاط
        if ($result) {
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'delete', 'user', $id, $user);
        }
        
        return $result;
    }
    
    public function getTotalCount() {
        $sql = "SELECT COUNT(*) as total FROM users WHERE is_active = 1";
        $stmt = $this->db->query($sql);
        return $stmt->fetch()['total'];
    }
    
    private function logActivity($userId, $action, $entityType, $entityId, $oldValues = null, $newValues = null) {
        try {
            $sql = "INSERT INTO activity_logs 
                    (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            
            $this->db->query($sql, [
                $userId,
                $action,
                $entityType,
                $entityId,
                $oldValues ? json_encode($oldValues, JSON_UNESCAPED_UNICODE) : null,
                $newValues ? json_encode($newValues, JSON_UNESCAPED_UNICODE) : null,
                $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ]);
        } catch (Exception $e) {
            error_log("خطأ في تسجيل النشاط: " . $e->getMessage());
        }
    }
}
?>