<?php
// models/StockTransactionModel.php
require_once 'core/Database.php';

class StockTransactionModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function recordInvoiceStock($invoiceId, $items) {
        $this->db->beginTransaction();
        
        try {
            foreach ($items as $item) {
                $sql = "INSERT INTO stock_transactions (item_id, type, quantity, reference_type, reference_id, created_at) 
                        VALUES (?, 'out', ?, 'invoice', ?, NOW())";
                
                $this->db->query($sql, [
                    $item['item_id'],
                    $item['quantity'],
                    $invoiceId
                ]);
                
                // تحديث المخزون
                $this->updateItemStock($item['item_id'], -$item['quantity']);
            }
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function getItemStockHistory($itemId, $page = 1, $perPage = 20) {
        $offset = ($page - 1) * $perPage;
        
        $sql = "SELECT st.*, 
                CASE 
                    WHEN st.reference_type = 'invoice' THEN i.invoice_number
                    ELSE 'تعديل'
                END as reference_number
                FROM stock_transactions st 
                LEFT JOIN invoices i ON st.reference_type = 'invoice' AND st.reference_id = i.id 
                WHERE st.item_id = ? 
                ORDER BY st.created_at DESC 
                LIMIT :limit OFFSET :offset";
        
        $stmt = $this->db->getConnection()->prepare($sql);
        $stmt->bindValue(':limit', (int)$perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute([$itemId]);
        
        return $stmt->fetchAll();
    }
    
    public function adjustStock($itemId, $quantity, $notes = '') {
        $this->db->beginTransaction();
        
        try {
            $type = $quantity > 0 ? 'in' : 'out';
            $absoluteQuantity = abs($quantity);
            
            $sql = "INSERT INTO stock_transactions (item_id, type, quantity, reference_type, reference_id, notes, created_at) 
                    VALUES (?, ?, ?, 'adjustment', 0, ?, NOW())";
            
            $this->db->query($sql, [
                $itemId,
                $type,
                $absoluteQuantity,
                $notes
            ]);
            
            // تحديث المخزون
            $this->updateItemStock($itemId, $quantity);
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    private function updateItemStock($itemId, $quantityChange) {
        $sql = "UPDATE items SET current_stock = current_stock + ?, updated_at = NOW() WHERE id = ?";
        $this->db->query($sql, [$quantityChange, $itemId]);
    }
    
    public function getTotalCount($itemId) {
        $sql = "SELECT COUNT(*) as total FROM stock_transactions WHERE item_id = ?";
        $stmt = $this->db->query($sql, [$itemId]);
        return $stmt->fetch()['total'];
    }
}
?>