<?php
// models/CustomerModel.php
require_once 'core/Database.php';

class CustomerModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    // داخل CustomerModel.php
public function getByName($name, $excludeId = null) {
    $db = Database::getInstance()->getConnection();

    $sql = "SELECT * FROM customers WHERE name = ?";
    $params = [$name];

    // إذا كان هناك ID للتعديل، نتجاهل هذا العميل نفسه
    if ($excludeId !== null) {
        $sql .= " AND id != ?";
        $params[] = $excludeId;
    }

    $stmt = $db->prepare($sql);
    $stmt->execute($params);

    return $stmt->fetch(); // يعيد صف واحد إذا وجد، أو false إذا لم يوجد
}

    
 public function getAll($page = 1, $perPage = 10, $search = '') {
    $offset = ($page - 1) * $perPage;

    $where = "WHERE is_active = 1";
    $params = [];

    if ($search !== '') {
        $where .= " AND (name LIKE :search1 OR phone LIKE :search2)";
        $params[':search1'] = "%{$search}%";
        $params[':search2'] = "%{$search}%";
    }

    $sql = "
        SELECT *
        FROM customers
        {$where}
        ORDER BY name ASC
        LIMIT {$perPage} OFFSET {$offset}
    ";

    $stmt = $this->db->getConnection()->prepare($sql);
    $stmt->execute($params);

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

    
    public function getById($id) {
        $sql = "SELECT * FROM customers WHERE id = ? AND is_active = 1";
        $stmt = $this->db->query($sql, [$id]);
        return $stmt->fetch();
    }
    
    public function create($data) {
        $this->db->beginTransaction();
        
        try {
            $sql = "INSERT INTO customers (name, phone, address, opening_balance, current_balance, created_at) 
                    VALUES (?, ?, ?, ?, ?, NOW())";
            
            $this->db->query($sql, [
                $data['name'],
                $data['phone'],
                $data['address'],
                $data['opening_balance'],
                $data['opening_balance'] // الرصيد الحالي يساوي الافتتاحي في البداية
            ]);
            
            $customerId = $this->db->lastInsertId();
            
            // تسجيل الحركة المالية للرصيد الافتتاحي
            if ($data['opening_balance'] > 0) {
                $this->recordOpeningBalanceTransaction($customerId, $data['opening_balance']);
            }
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'create', 'customer', $customerId, null, $data);
            
            $this->db->commit();
            
            return $customerId;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function update($id, $data) {
        $this->db->beginTransaction();
        
        try {
            $oldCustomer = $this->getById($id);
            
            $sql = "UPDATE customers SET name = ?, phone = ?, address = ?, updated_at = NOW() 
                    WHERE id = ?";
            
            $this->db->query($sql, [
                $data['name'],
                $data['phone'],
                $data['address'],
                $id
            ]);
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'update', 'customer', $id, $oldCustomer, $data);
            
            $this->db->commit();
            
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function delete($id) {
        $customer = $this->getById($id);
        
        $sql = "UPDATE customers SET is_active = 0, updated_at = NOW() WHERE id = ?";
        $result = $this->db->query($sql, [$id]);
        
        // تسجيل النشاط
        if ($result) {
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'delete', 'customer', $id, $customer);
        }
        
        return $result;
    }
    
    public function getTotalCount($search = '') {
        $where = [];
        $params = [];
        
        if (!empty($search)) {
            $where[] = "(name LIKE ? OR phone LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        $where[] = "is_active = 1";
        $whereClause = "WHERE " . implode(" AND ", $where);
        
        $sql = "SELECT COUNT(*) as total FROM customers {$whereClause}";
        $stmt = $this->db->query($sql, $params);
        return $stmt->fetch()['total'];
    }
    
    private function recordOpeningBalanceTransaction($customerId, $amount) {
        $sql = "INSERT INTO transactions 
                (date, account_type, account_id, debit_amount, credit_amount, description, balance_after, created_at) 
                VALUES (CURDATE(), 'customer', ?, ?, 0, 'رصيد افتتاحي', ?, NOW())";
        
        $this->db->query($sql, [
            $customerId,
            $amount,
            $amount
        ]);
    }
    
    private function logActivity($userId, $action, $entityType, $entityId, $oldValues = null, $newValues = null) {
        try {
            $sql = "INSERT INTO activity_logs 
                    (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            
            $this->db->query($sql, [
                $userId,
                $action,
                $entityType,
                $entityId,
                $oldValues ? json_encode($oldValues, JSON_UNESCAPED_UNICODE) : null,
                $newValues ? json_encode($newValues, JSON_UNESCAPED_UNICODE) : null,
                $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ]);
        } catch (Exception $e) {
            error_log("خطأ في تسجيل النشاط: " . $e->getMessage());
        }
    }
    // في نهاية models/CustomerModel.php - أضف هذه الدوال:

    public function getAllActive() {
        $sql = "SELECT id, name, phone FROM customers WHERE is_active = 1 ORDER BY name";
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }
    
    // دالة إضافية للحصول على العملاء مع أرصدتهم
    public function getAllWithBalance() {
        $sql = "SELECT id, name, phone, current_balance FROM customers WHERE is_active = 1 ORDER BY name";
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }
}
?>