<?php
// التأكد من عدم وجود أي إخراج قبل الرؤوس
ob_start();

require 'vendor/autoload.php';
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;
use PhpOffice\PhpSpreadsheet\Worksheet\PageSetup;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

require_once 'config.php';
$db = Config::getDB();

function format_currency_excel($amount, $symbol) {
    return number_format($amount, 2) . ' ' . $symbol;
}

/* =======================
   دالة تحويل رقم إلى كلمات
======================= */
function numberToWords($number) {
    $hyphen = '-';
    $conjunction = ' and ';
    $separator = ', ';
    $negative = 'negative ';
    $dictionary = [
        0 => 'zero',
        1 => 'one',
        2 => 'two',
        3 => 'three',
        4 => 'four',
        5 => 'five',
        6 => 'six',
        7 => 'seven',
        8 => 'eight',
        9 => 'nine',
        10 => 'ten',
        11 => 'eleven',
        12 => 'twelve',
        13 => 'thirteen',
        14 => 'fourteen',
        15 => 'fifteen',
        16 => 'sixteen',
        17 => 'seventeen',
        18 => 'eighteen',
        19 => 'nineteen',
        20 => 'twenty',
        30 => 'thirty',
        40 => 'forty',
        50 => 'fifty',
        60 => 'sixty',
        70 => 'seventy',
        80 => 'eighty',
        90 => 'ninety',
        100 => 'hundred',
        1000 => 'thousand'
    ];

    if (!is_numeric($number)) return false;
    if ($number < 0) return $negative . numberToWords(abs($number));

    $string = '';
    if ($number < 21) {
        $string = $dictionary[$number];
    } elseif ($number < 100) {
        $tens = ((int) ($number / 10)) * 10;
        $units = $number % 10;
        $string = $dictionary[$tens];
        if ($units) {
            $string .= $hyphen . $dictionary[$units];
        }
    } elseif ($number < 1000) {
        $hundreds = (int) ($number / 100);
        $remainder = $number % 100;
        $string = $dictionary[$hundreds] . ' ' . $dictionary[100];
        if ($remainder) {
            $string .= $conjunction . numberToWords($remainder);
        }
    } else {
        $thousands = (int) ($number / 1000);
        $remainder = $number % 1000;
        $string = numberToWords($thousands) . ' ' . $dictionary[1000];
        if ($remainder) {
            if ($remainder < 100) $string .= $conjunction;
            else $string .= $separator;
            $string .= numberToWords($remainder);
        }
    }
    return $string;
}

/* =======================
      جلب رقم الفاتورة
======================= */
$invoice_id = $_GET['invoice_id'] ?? null;
$currency_id = $_GET['currency_id'] ?? null;

if (!$invoice_id || !$currency_id) {
    ob_end_clean();
    die('رقم الفاتورة والعملة مطلوبان');
}

/* =======================
     جلب بيانات الفاتورة
======================= */
$stmt = $db->prepare("
    SELECT i.*, 
           c.name AS customer_name,
           c.phone AS customer_phone,
           c.address AS customer_address,
           cur.code AS currency_code,
           cur.name AS currency_name
    FROM invoices i
    JOIN customers c ON i.customer_id = c.id
    JOIN currencies cur ON i.currency_id = cur.id
    WHERE i.id = :id
");
$stmt->execute([':id' => $invoice_id]);
$invoice = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$invoice) {
    ob_end_clean();
    die('الفاتورة غير موجودة');
}

/* =======================
   جلب سعر الصرف للفاتورة
======================= */
$stmt_rate = $db->prepare("
    SELECT 
        ier.exchange_rate,
        cur.code,
        cur.name,
        cur.symbol
    FROM invoice_exchange_rates ier
    JOIN currencies cur ON ier.currency_id = cur.id
    WHERE ier.invoice_id = :invoice_id
      AND ier.currency_id = :currency_id
    LIMIT 1
");
$stmt_rate->execute([
    ':invoice_id' => $invoice_id,
    ':currency_id' => $currency_id
]);
$exchange = $stmt_rate->fetch(PDO::FETCH_ASSOC);

if (!$exchange) {
    ob_end_clean();
    die('لا يوجد سعر صرف مقيد لهذه الفاتورة مع العملة المختارة');
}

$exchange_rate   = (float)$exchange['exchange_rate'];
$print_currency  = $exchange['code'];
$currency_name   = $exchange['name'];
$currency_symbol = $exchange['symbol'];

/* =======================
      جلب عناصر الفاتورة
======================= */
$stmt_items = $db->prepare("
    SELECT ii.*, 
           it.name AS item_name,
           it.model AS item_model,
           it.specifications AS item_specifications,
           ii.unit_price AS quoted_price_per_unit,
           it.image AS item_image,
           comp.name AS company_name,
           u.code AS unit_name,
           ct.total_cartons AS ctn_value,
           w.total_weight AS weight_value,
           cb.total_cbm AS cbm_value
    FROM invoice_items ii
    JOIN items it ON ii.item_id = it.id
    LEFT JOIN companies comp ON ii.company_id = comp.id
    LEFT JOIN units u ON it.unit_id = u.id
    LEFT JOIN shared_cartons ct ON ii.carton_id = ct.id
    LEFT JOIN shared_weights w ON ii.weight_id = w.id
    LEFT JOIN shared_cbm cb ON ii.cbm_id = cb.id
    WHERE ii.invoice_id = :invoice_id
    ORDER BY ii.created_at
");
$stmt_items->execute([':invoice_id' => $invoice_id]);
$invoice_items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);

if (!$invoice_items) {
    ob_end_clean();
    die('لا توجد عناصر في هذه الفاتورة');
}

/* =======================
        حساب الإجماليات
======================= */
$grand_total = 0;
foreach ($invoice_items as $item) {
    $converted_price = $item['quoted_price_per_unit'] * $exchange_rate;
    $item_total = $item['quantity'] * $converted_price;
    $grand_total += $item_total;
}

$commission_percent = (float)($invoice['office_commission'] ?? 0);
$commission_value   = ($grand_total * $commission_percent) / 100;
$certificates_cost  = ((float)($invoice['certificate_costs'] ?? 0)) * $exchange_rate;
$final_total = $grand_total + $commission_value + $certificates_cost;

/* =======================
   إنشاء ملف Excel
======================= */
$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();

// إعداد حجم الورقة A4
$sheet->getPageSetup()->setPaperSize(PageSetup::PAPERSIZE_A4);
$sheet->getPageSetup()->setOrientation(PageSetup::ORIENTATION_PORTRAIT);
$sheet->getPageSetup()->setFitToPage(true);
$sheet->getPageSetup()->setFitToWidth(1);
$sheet->getPageSetup()->setFitToHeight(0);

// إعداد الهوامش (مطابقة للطباعة)
$sheet->getPageMargins()->setTop(0.1);
$sheet->getPageMargins()->setRight(0.3);
$sheet->getPageMargins()->setLeft(0.3);
$sheet->getPageMargins()->setBottom(0.1);

// إعداد اسم الورقة
$sheet->setTitle('Invoice');

// إعداد اتجاه النص من اليسار لليمين (مطابقة للطباعة)
$sheet->setRightToLeft(false);

// إعداد الخط (Times New Roman مثل الطباعة)
$defaultFont = new Font();
$defaultFont->setName('Times New Roman');
$defaultFont->setSize(10);
$spreadsheet->getDefaultStyle()->applyFromArray([
    'font' => [
        'name' => 'Times New Roman',
        'size' => 10,
    ]
]);

// إعداد أبعاد الأعمدة (مطابقة للتصميم)
$sheet->getColumnDimension('A')->setWidth(2);      // No.
$sheet->getColumnDimension('B')->setWidth(12);     // Model No.
$sheet->getColumnDimension('C')->setWidth(20);     // Name
$sheet->getColumnDimension('D')->setWidth(12);     // Picture
$sheet->getColumnDimension('E')->setWidth(21);     // Specification
$sheet->getColumnDimension('F')->setWidth(10);     // Price
$sheet->getColumnDimension('G')->setWidth(5);      // Quantity
$sheet->getColumnDimension('H')->setWidth(5);      // Unit
$sheet->getColumnDimension('I')->setWidth(12);     // Total

/* =======================
   إضافة اللوجو الرئيسي (امتداد كامل)
======================= */
$currentRow = 1;
$logoPath = 'logo2.png';
if (file_exists($logoPath)) {
    $drawing = new Drawing();
    $drawing->setName('Company Logo');
    $drawing->setDescription('Logo');
    $drawing->setPath($logoPath);
    $drawing->setCoordinates('A' . $currentRow);
    $drawing->setWidth(110);
    $drawing->setHeight(110);
    $drawing->setOffsetX(0);
    $drawing->setOffsetY(0);
    $drawing->setWorksheet($sheet);
    
    // دمج الخلايا للوجو ليمتد على عرض الصفحة
    $sheet->mergeCells('A' . $currentRow . ':F' . ($currentRow + 4));
    $sheet->getRowDimension($currentRow)->setRowHeight(100);
    $sheet->getRowDimension($currentRow + 1)->setRowHeight(0);
    $sheet->getRowDimension($currentRow + 2)->setRowHeight(0);
    $sheet->getRowDimension($currentRow + 3)->setRowHeight(0);
    $sheet->getRowDimension($currentRow + 4)->setRowHeight(0);
    
    $currentRow += 5;
}

/* =======================
   إضافة صورة الهاتف
======================= */
$phoneLogoPath = 'phone.png';
if (file_exists($phoneLogoPath)) {
    $drawing2 = new Drawing();
    $drawing2->setName('Phone Logo');
    $drawing2->setDescription('Phone');
    $drawing2->setPath($phoneLogoPath);
    $drawing2->setCoordinates('B' . $currentRow);
    $drawing2->setWidth(150);
    $drawing2->setHeight(60);
    $drawing2->setOffsetX(10);
    $drawing2->setOffsetY(5);
    $drawing2->setWorksheet($sheet);
    
    $sheet->mergeCells('B' . $currentRow . ':H' . ($currentRow + 2));
    $sheet->getRowDimension($currentRow)->setRowHeight(20);
    $sheet->getRowDimension($currentRow + 1)->setRowHeight(20);
    $sheet->getRowDimension($currentRow + 2)->setRowHeight(20);
    
    $currentRow += 3;
}

/* =======================
   الخط الفاصل
======================= */
$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, '--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------');
$sheet->getStyle('A' . $currentRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
$sheet->getRowDimension($currentRow)->setRowHeight(5);
$currentRow++;

/* =======================
   معلومات الفاتورة (مطابقة للتصميم بالضبط)
======================= */
// الصف الأول: معلومات الفاتورة
$sheet->setCellValue('A' . $currentRow, 'Invoice NO: ' . $invoice['invoice_number']);
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF'], 'size' => 10],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'EAA033']],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER]
]);
$sheet->getColumnDimension('A')->setAutoSize(false);
$sheet->getColumnDimension('A')->setWidth(20);

$sheet->setCellValue('E' . $currentRow, 'INVOICE');
$sheet->getStyle('E' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 14],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER]
]);

$sheet->setCellValue('I' . $currentRow, 'To: ' . $invoice['customer_name']);
$sheet->getStyle('I' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF'], 'size' => 10],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'EAA033']],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER]
]);

// محاذاة الخلايا
$sheet->mergeCells('A' . $currentRow . ':B' . $currentRow);
$sheet->mergeCells('E' . $currentRow . ':F' . $currentRow);
$sheet->mergeCells('I' . $currentRow . ':I' . $currentRow);

$currentRow++;

// الصف الثاني: التاريخ والعملة
$sheet->setCellValue('A' . $currentRow, 'Date: ' . date('Y-m-d', strtotime($invoice['created_at'])));
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 10],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_LEFT]
]);

$sheet->setCellValue('I' . $currentRow, 'Currency: ' . $currency_name . ' (' . $currency_symbol . ')');
$sheet->getStyle('I' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 10],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_LEFT]
]);

$currentRow++;
$currentRow++; // سطر فارغ

/* =======================
   رؤوس الجدول (مطابقة للتصميم)
======================= */
$headerRow = $currentRow;
$headers = [
    ['cell' => 'A', 'text' => 'No.', 'color' => '22445D', 'font_color' => '000000', 'width' => 2],
    ['cell' => 'B', 'text' => 'Model No.', 'color' => 'EAA033', 'font_color' => '000000', 'width' => 12],
    ['cell' => 'C', 'text' => 'Name', 'color' => '22445D', 'font_color' => '000000', 'width' => 20],
    ['cell' => 'D', 'text' => 'Picture', 'color' => 'EAA033', 'font_color' => '000000', 'width' => 12],
    ['cell' => 'E', 'text' => 'Specification', 'color' => '22445D', 'font_color' => '000000', 'width' => 21],
    ['cell' => 'F', 'text' => 'Price', 'color' => 'EAA033', 'font_color' => '000000', 'width' => 10],
    ['cell' => 'G', 'text' => 'Quantity', 'color' => '22445D', 'font_color' => '000000', 'width' => 10],
    ['cell' => 'I', 'text' => 'Total', 'color' => 'EAA033', 'font_color' => '000000', 'width' => 12]
];

foreach ($headers as $header) {
    $cell = $header['cell'] . $headerRow;
    $sheet->setCellValue($cell, $header['text']);
    
    $styleArray = [
        'font' => [
            'bold' => true,
            'size' => 10,
            'name' => 'Times New Roman',
            'color' => ['rgb' => $header['font_color']]
        ],
        'alignment' => [
            'horizontal' => Alignment::HORIZONTAL_CENTER,
            'vertical' => Alignment::VERTICAL_CENTER,
            'wrapText' => true
        ],
        'borders' => [
            'allBorders' => [
                'borderStyle' => Border::BORDER_THIN,
                'color' => ['rgb' => '000000']
            ]
        ]
    ];
    
    if ($header['color'] === 'FFFF00') {
        $styleArray['fill'] = [
            'fillType' => Fill::FILL_SOLID,
            'startColor' => ['rgb' => $header['color']]
        ];
        $styleArray['font']['color'] = ['rgb' => '000000'];
    } else {
        $styleArray['fill'] = [
            'fillType' => Fill::FILL_SOLID,
            'startColor' => ['rgb' => $header['color']]
        ];
        $styleArray['font']['color'] = ['rgb' => 'FFFFFF'];
    }
    
    $sheet->getStyle($cell)->applyFromArray($styleArray);
    
    // دمج خلايا الكمية والوحدة
    if ($header['cell'] == 'G') {
        $sheet->mergeCells('G' . $headerRow . ':H' . $headerRow);
        $sheet->setCellValue('G' . $headerRow, 'Quantity');
    }
}

// إعداد ارتفاع صف الرؤوس
$sheet->getRowDimension($headerRow)->setRowHeight(25);
$currentRow++;

/* =======================
   بيانات العناصر
======================= */
$counter = 1;
foreach ($invoice_items as $item) {
    $converted_price = $item['quoted_price_per_unit'] * $exchange_rate;
    $item_total = $item['quantity'] * $converted_price;
    
    // إعداد ارتفاع الصف للصورة
    $sheet->getRowDimension($currentRow)->setRowHeight(80);
    
    // No.
    $sheet->setCellValue('A' . $currentRow, $counter++);
    $sheet->getStyle('A' . $currentRow)->applyFromArray([
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
        'font' => ['size' => 10, 'name' => 'Times New Roman']
    ]);
    
    // Model No.
    $sheet->setCellValue('B' . $currentRow, $item['item_model']);
    $sheet->getStyle('B' . $currentRow)->applyFromArray([
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
        'font' => ['size' => 9, 'name' => 'Times New Roman']
    ]);
    
    // Name
    $sheet->setCellValue('C' . $currentRow, $item['item_name']);
    $sheet->getStyle('C' . $currentRow)->applyFromArray([
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
        'font' => ['bold' => true, 'size' => 10, 'name' => 'Times New Roman']
    ]);
    
    // Picture
    if (!empty($item['item_image'])) {
        $imagePath = 'uploads/items/' . $item['item_image'];
        if (file_exists($imagePath)) {
            $drawing = new Drawing();
            $drawing->setName('Item Image ' . ($counter-1));
            $drawing->setDescription('Item Image');
            $drawing->setPath($imagePath);
            $drawing->setCoordinates('D' . $currentRow);
            $drawing->setWidth(80);
            $drawing->setHeight(60);
            $drawing->setOffsetX(5);
            $drawing->setOffsetY(5);
            $drawing->setWorksheet($sheet);
        }
    }
    
    $sheet->getStyle('D' . $currentRow)->applyFromArray([
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]]
    ]);
    
    // Specification - إصلاح مشكلة عرض النص الكامل
    $specText = $item['item_specifications'];
    $sheet->setCellValue('E' . $currentRow, $specText);
    $sheet->getStyle('E' . $currentRow)->applyFromArray([
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_LEFT, 'vertical' => Alignment::VERTICAL_TOP, 'wrapText' => true],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
        'font' => ['size' => 8, 'name' => 'Times New Roman']
    ]);
    $sheet->getRowDimension($currentRow)->setRowHeight(80); // ارتفاع مناسب للنص الطويل
    
    // Price
    $sheet->setCellValue('F' . $currentRow, format_currency_excel($converted_price, $currency_symbol));
    $sheet->getStyle('F' . $currentRow)->applyFromArray([
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
        'font' => ['size' => 10, 'name' => 'Times New Roman']
    ]);
    
    // Quantity
    $sheet->setCellValue('G' . $currentRow, number_format($item['quantity']));
    $sheet->getStyle('G' . $currentRow)->applyFromArray([
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
        'font' => ['size' => 10, 'name' => 'Times New Roman']
    ]);
    
    // Unit
    $sheet->setCellValue('H' . $currentRow, $item['unit_name']);
    $sheet->getStyle('H' . $currentRow)->applyFromArray([
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
        'font' => ['size' => 10, 'name' => 'Times New Roman']
    ]);
    
    // دمج خلايا الكمية والوحدة
    $sheet->mergeCells('G' . $currentRow . ':H' . $currentRow);
    
    // Total
    $sheet->setCellValue('I' . $currentRow, format_currency_excel($item_total, $currency_symbol));
    $sheet->getStyle('I' . $currentRow)->applyFromArray([
        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
        'font' => ['size' => 10, 'name' => 'Times New Roman']
    ]);
    
    $currentRow++;
}

/* =======================
   الإجماليات
======================= */
// Total
$sheet->mergeCells('A' . $currentRow . ':H' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Total');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'name' => 'Times New Roman'],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
    'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F8F9FA']]
]);

$sheet->setCellValue('I' . $currentRow, format_currency_excel($grand_total, $currency_symbol));
$sheet->getStyle('I' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'color' => ['rgb' => 'EAA033'], 'name' => 'Times New Roman'],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F0F0F0']],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
    'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]]
]);

$currentRow++;

// Commission
$sheet->mergeCells('A' . $currentRow . ':H' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Commission "' . $commission_percent . '%"');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'name' => 'Times New Roman'],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
    'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F8F9FA']]
]);

$sheet->setCellValue('I' . $currentRow, format_currency_excel($commission_value, $currency_symbol));
$sheet->getStyle('I' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'color' => ['rgb' => 'EAA033'], 'name' => 'Times New Roman'],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F0F0F0']],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
    'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]]
]);

$currentRow++;

// Certificates Cost
$sheet->mergeCells('A' . $currentRow . ':H' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Certificates Cost');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'name' => 'Times New Roman'],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
    'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F8F9FA']]
]);

$sheet->setCellValue('I' . $currentRow, format_currency_excel($certificates_cost, $currency_symbol));
$sheet->getStyle('I' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'color' => ['rgb' => 'EAA033'], 'name' => 'Times New Roman'],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F0F0F0']],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
    'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]]
]);

$currentRow++;

// Final Total
$sheet->mergeCells('A' . $currentRow . ':H' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Final Total');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'name' => 'Times New Roman'],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
    'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F8F9FA']]
]);

$sheet->setCellValue('I' . $currentRow, format_currency_excel($final_total, $currency_symbol));
$sheet->getStyle('I' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'color' => ['rgb' => 'EAA033'], 'name' => 'Times New Roman'],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F0F0F0']],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
    'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]]
]);

$currentRow++;

// Amount in Words
$currency_map = [
    'الرينمنبي' => 'Chinese Yuan',
    'الدولار الأمريكي' => 'US Dollar',
    'الريال السعودي' => 'Saudi Riyal'
];
$currency_en = $currency_map[$currency_name] ?? $currency_name;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, $currency_en . ' ' . ucfirst(numberToWords((int)$final_total)) . ' Only');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'color' => ['rgb' => 'EAA033'], 'name' => 'Times New Roman'],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER]
]);

$currentRow += 2;

/* =======================
   الملاحظات
======================= */
// Remark
$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Remark:');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, '1. Price term: EXW, not including any freight or other cost');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, '2. Delivery time: 25-30 workdays against 30% deposit');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['size' => 10, 'name' => 'Times New Roman']
]);

$currentRow += 2;

/* =======================
   معلومات التحويل البنكي
======================= */
// Remitting Route
$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'REMITTING ROUTE FOR USD (美元汇款路线)');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 11, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Intermediary Bank:');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'JPMORGAN CHASE BANK, N.A. NEW YORK');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, '(中间行 54/56A)');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['size' => 9, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'SWIFT BIC: CHASUS33XXX');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Beneficiary\'s Bank:');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'ZHEJIANG CHOUZHOU COMMERCIAL BANK CO., LTD.');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'SWIFT BIC: CZCBCN2X');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Beneficiary: TWO STARS EQUIPMENT AND TRADE CO., LIMITED');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Account Number: NRA15622142010500005063');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'Account Address: ZONE 2, INTERNATIONAL TRADE CITY, NO. 1180,');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['bold' => true, 'size' => 10, 'name' => 'Times New Roman']
]);

$currentRow++;

$sheet->mergeCells('A' . $currentRow . ':I' . $currentRow);
$sheet->setCellValue('A' . $currentRow, 'NORTH CHONGZHOU ROAD, YIWU CITY, JINHUA, ZHEJIANG PROVIANCE, CHINA');
$sheet->getStyle('A' . $currentRow)->applyFromArray([
    'font' => ['size' => 10, 'name' => 'Times New Roman']
]);

/* =======================
   إضافة صورة الختم
======================= */
$stampPath = 'stamp.png';
if (file_exists($stampPath)) {
    $stampRow = $currentRow - 12; // وضع الختم في المكان المناسب
    
    $drawing3 = new Drawing();
    $drawing3->setName('Stamp');
    $drawing3->setDescription('Stamp');
    $drawing3->setPath($stampPath);
    $drawing3->setCoordinates('G' . $stampRow);
    $drawing3->setWidth(120);
    $drawing3->setHeight(80);
    $drawing3->setOffsetX(10);
    $drawing3->setOffsetY(10);
    $drawing3->setWorksheet($sheet);
}

/* =======================
   إضافة صورة التذييل (امتداد كامل مثل الأعلى)
======================= */
$footerPath = 'stylefoot.png';
if (file_exists($footerPath)) {
    $footerRow = $currentRow + 1;
    
    $drawing4 = new Drawing();
    $drawing4->setName('Footer');
    $drawing4->setDescription('Footer');
    $drawing4->setPath($footerPath);
    $drawing4->setCoordinates('A' . $footerRow);
    $drawing4->setWidth(100);
    $drawing4->setHeight(100);
    $drawing4->setOffsetX(0);
    $drawing4->setOffsetY(0);
    $drawing4->setWorksheet($sheet);
    
    $sheet->mergeCells('A' . $footerRow . ':I' . ($footerRow + 2));
    $sheet->getRowDimension($footerRow)->setRowHeight(40);
    $sheet->getRowDimension($footerRow + 1)->setRowHeight(0);
    $sheet->getRowDimension($footerRow + 2)->setRowHeight(0);
}

/* =======================
   إعدادات نهائية للطباعة
======================= */
// إخفاء خطوط الشبكة
$sheet->setShowGridlines(false);

// إعداد الطباعة
$sheet->getPageSetup()->setHorizontalCentered(true);
$sheet->getPageSetup()->setVerticalCentered(false);

/* =======================
   إرسال الملف
======================= */
// تنظيف المخزن المؤقت
ob_end_clean();

// تعيين رؤوس HTTP
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment;filename="invoice_' . $invoice['invoice_number'] . '.xlsx"');
header('Cache-Control: max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

// حفظ وإرسال الملف
$writer = new Xlsx($spreadsheet);

// استخدام ملف مؤقت للتأكد من سلامة الملف
$tempFile = tempnam(sys_get_temp_dir(), 'excel_') . '.xlsx';
$writer->save($tempFile);

// إرسال الملف
if (file_exists($tempFile)) {
    header('Content-Length: ' . filesize($tempFile));
    readfile($tempFile);
    unlink($tempFile);
}

exit;