<?php
// create_backup.php
session_start();
require_once 'config/database.php';
require_once 'core/Database.php';
require_once 'core/Session.php';
require_once 'core/Auth.php';

$session = new Session();
$auth = new Auth();

if (!$auth->isLoggedIn()) {
    header('HTTP/1.1 403 Forbidden');
    exit('غير مصرح لك بتنفيذ هذه العملية');
}

// تعريف المجلدات
$backupDir = __DIR__ . '/backups';
$uploadsDir = __DIR__ . '/uploads';
$backupFile = $backupDir . '/backup_' . date('Y-m-d_H-i-s') . '.zip';

// إنشاء مجلد النسخ الاحتياطية إذا لم يكن موجوداً
if (!file_exists($backupDir)) {
    if (!mkdir($backupDir, 0755, true)) {
        header('HTTP/1.1 500 Internal Server Error');
        exit('لا يمكن إنشاء مجلد النسخ الاحتياطية');
    }
}

// التحقق من أن إضافة ZIP متوفرة
if (!class_exists('ZipArchive')) {
    header('HTTP/1.1 500 Internal Server Error');
    exit('إضافة ZIP غير متوفرة في الخادم. يرجى تفعيلها.');
}

// إنشاء كائن ZIP
$zip = new ZipArchive();
if ($zip->open($backupFile, ZipArchive::CREATE) !== TRUE) {
    header('HTTP/1.1 500 Internal Server Error');
    exit('لا يمكن إنشاء ملف النسخة الاحتياطية');
}

// إضافة ملف SQL لقاعدة البيانات
try {
    $db = Database::getInstance();
    
    // جلب جميع الجداول
    $tables = $db->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
    
    $sqlContent = "-- نسخة احتياطية لنظام GTS\n";
    $sqlContent .= "-- تاريخ الإنشاء: " . date('Y-m-d H:i:s') . "\n";
    $sqlContent .= "-- =============================================\n\n";
    
    $tablesCount = 0;
    $rowsCount = 0;
    
    foreach ($tables as $table) {
        $tablesCount++;
        
        // جلب هيكل الجدول
        $createTable = $db->query("SHOW CREATE TABLE `$table`")->fetch();
        if ($createTable) {
            $sqlContent .= "--\n-- هيكل الجدول: $table\n--\n";
            $sqlContent .= "DROP TABLE IF EXISTS `$table`;\n";
            $sqlContent .= $createTable['Create Table'] . ";\n\n";
        }
        
        // جلب البيانات
        try {
            $rows = $db->query("SELECT * FROM `$table`")->fetchAll(PDO::FETCH_ASSOC);
            $rowsCount += count($rows);
            
            if (count($rows) > 0) {
                $sqlContent .= "--\n-- بيانات الجدول: $table (" . count($rows) . " سجل)\n--\n";
                
                // جلب أسماء الأعمدة
                $columns = array_keys($rows[0]);
                
                foreach ($rows as $row) {
                    $values = [];
                    foreach ($row as $key => $value) {
                        if (is_null($value)) {
                            $values[] = 'NULL';
                        } elseif (is_numeric($value)) {
                            $values[] = $value;
                        } else {
                            // Escape القيم النصية
                            $values[] = "'" . addslashes($value) . "'";
                        }
                    }
                    
                    $sqlContent .= "INSERT INTO `$table` (`" . implode('`, `', $columns) . "`) VALUES(" . implode(', ', $values) . ");\n";
                }
                $sqlContent .= "\n";
            }
        } catch (Exception $e) {
            $sqlContent .= "-- خطأ في جلب بيانات الجدول $table: " . $e->getMessage() . "\n\n";
        }
    }
    
    // إضافة ملف SQL إلى الأرشيف
    if (!$zip->addFromString('database_backup.sql', $sqlContent)) {
        throw new Exception('فشل في إضافة ملف SQL إلى الأرشيف');
    }
    
} catch (Exception $e) {
    $zip->close();
    if (file_exists($backupFile)) {
        unlink($backupFile);
    }
    header('HTTP/1.1 500 Internal Server Error');
    exit('خطأ في تصدير قاعدة البيانات: ' . $e->getMessage());
}

// إضافة مجلد uploads إذا كان موجوداً
$filesAdded = 0;
if (file_exists($uploadsDir) && is_dir($uploadsDir)) {
    try {
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($uploadsDir),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $name => $file) {
            if (!$file->isDir()) {
                $filePath = $file->getRealPath();
                $relativePath = substr($filePath, strlen(__DIR__) + 1);
                
                if ($zip->addFile($filePath, $relativePath)) {
                    $filesAdded++;
                }
            }
        }
    } catch (Exception $e) {
        // لا توقف العملية إذا حدث خطأ في إضافة الملفات
        error_log("خطأ في إضافة ملفات uploads: " . $e->getMessage());
    }
}

// إضافة ملف معلومات النظام
$infoContent = "معلومات النسخة الاحتياطية\n";
$infoContent .= "=========================\n";
$infoContent .= "النظام: GTS لمعدات المطبخ المحدودة\n";
$infoContent .= "تاريخ النسخة: " . date('Y-m-d H:i:s') . "\n";
$infoContent .= "المستخدم: " . ($_SESSION['username'] ?? 'غير معروف') . "\n";
$infoContent .= "نوع الخادم: " . $_SERVER['SERVER_SOFTWARE'] . "\n";
$infoContent .= "نسخة PHP: " . PHP_VERSION . "\n";
$infoContent .= "\nالمجموع:\n";
$infoContent .= "- جداول قاعدة البيانات: " . $tablesCount . "\n";
$infoContent .= "- السجلات المصدرة: " . $rowsCount . "\n";
$infoContent .= "- ملفات uploads: " . $filesAdded . "\n";
$infoContent .= "- حجم الأرشيف: " . filesize($backupFile) . " بايت\n";

$zip->addFromString('backup_info.txt', $infoContent);

// إغلاق الأرشيف
if (!$zip->close()) {
    header('HTTP/1.1 500 Internal Server Error');
    exit('فشل في إغلاق الأرشيف');
}

// التحقق من أن الملف تم إنشاؤه وله حجم
if (!file_exists($backupFile) || filesize($backupFile) == 0) {
    header('HTTP/1.1 500 Internal Server Error');
    exit('تم إنشاء الأرشيف لكنه فارغ');
}

// إرسال الملف للتحميل
header('Content-Type: application/zip');
header('Content-Disposition: attachment; filename="backup_gts_' . date('Y-m-d') . '.zip"');
header('Content-Length: ' . filesize($backupFile));
header('Cache-Control: no-cache, must-revalidate');
header('Expires: 0');
header('Pragma: public');

readfile($backupFile);

// حذف الملف المؤقت بعد الإرسال
// unlink($backupFile);
exit;