<?php
// controllers/UserController.php
require_once 'models/UserModel.php';
require_once 'models/RoleModel.php';
require_once 'models/ActivityLog.php';
require_once 'core/Auth.php';
require_once 'core/Session.php';

class UserController {
    private $userModel;
    private $roleModel;
    private $auth;
    private $session;
    
    public function __construct() {
        $this->userModel = new UserModel();
        $this->roleModel = new RoleModel();
        $this->auth = new Auth();
        $this->session = new Session();
        
        // التحقق من الصلاحيات
        if (!$this->auth->isLoggedIn()) {
            header('Location: login.php');
            exit;
        }
        
        if (!$this->auth->checkPermission('users', 'read')) {
            $this->session->setFlash('ليس لديك صلاحية للوصول إلى هذه الصفحة', 'error');
            header('Location: index.php');
            exit;
        }
    }
    
    public function index() {
        $page = $_GET['page'] ?? 1;
        $perPage = 10;
        
        $users = $this->userModel->getAll($page, $perPage);
        $totalUsers = $this->userModel->getTotalCount();
        $totalPages = ceil($totalUsers / $perPage);
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/users/index.php';
    }
    
    public function create() {
        if (!$this->auth->checkPermission('users', 'create')) {
            $this->session->setFlash('ليس لديك صلاحية لإضافة مستخدمين', 'error');
            header('Location: users.php');
            exit;
        }
        
        $roles = $this->roleModel->getAll();
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'username' => trim($_POST['username']),
                    'email' => trim($_POST['email']),
                    'password' => $_POST['password'],
                    'role_id' => (int)$_POST['role_id']
                ];
                
                // التحقق من صحة البيانات
                if (empty($data['username']) || empty($data['email']) || empty($data['password'])) {
                    throw new Exception('جميع الحقول مطلوبة');
                }
                
                if (strlen($data['password']) < 6) {
                    throw new Exception('كلمة المرور يجب أن تكون 6 أحرف على الأقل');
                }
                
                $userId = $this->userModel->create($data);
                
                $this->session->setFlash('تم إضافة المستخدم بنجاح', 'success');
                header('Location: users.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/users/create.php';
    }
    
    public function edit($id) {
        if (!$this->auth->checkPermission('users', 'update')) {
            $this->session->setFlash('ليس لديك صلاحية لتعديل المستخدمين', 'error');
            header('Location: users.php');
            exit;
        }
        
        $user = $this->userModel->getById($id);
        if (!$user) {
            $this->session->setFlash('المستخدم غير موجود', 'error');
            header('Location: users.php');
            exit;
        }
        
        $roles = $this->roleModel->getAll();
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'username' => trim($_POST['username']),
                    'email' => trim($_POST['email']),
                    'role_id' => (int)$_POST['role_id'],
                    'password' => $_POST['password'] // اختياري
                ];
                
                // التحقق من صحة البيانات
                if (empty($data['username']) || empty($data['email'])) {
                    throw new Exception('اسم المستخدم والبريد الإلكتروني مطلوبان');
                }
                
                $this->userModel->update($id, $data);
                
                $this->session->setFlash('تم تحديث المستخدم بنجاح', 'success');
                header('Location: users.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/users/edit.php';
    }
    
    public function delete($id) {
        if (!$this->auth->checkPermission('users', 'delete')) {
            $this->session->setFlash('ليس لديك صلاحية لحذف المستخدمين', 'error');
            header('Location: users.php');
            exit;
        }
        
        $user = $this->userModel->getById($id);
        if (!$user) {
            $this->session->setFlash('المستخدم غير موجود', 'error');
        } else {
            $this->userModel->delete($id);
            $this->session->setFlash('تم حذف المستخدم بنجاح', 'success');
        }
        
        header('Location: users.php');
        exit;
    }
}
?>