<?php
// controllers/UnitController.php
require_once 'models/UnitModel.php';
require_once 'core/Auth.php';
require_once 'core/Session.php';

class UnitController {
    private $unitModel;
    private $auth;
    private $session;
    
    public function __construct() {
        $this->unitModel = new UnitModel();
        $this->auth = new Auth();
        $this->session = new Session();
        
        // التحقق من الصلاحيات
        if (!$this->auth->isLoggedIn()) {
            header('Location: login.php');
            exit;
        }
        
        if (!$this->auth->checkPermission('units', 'read')) {
            $this->session->setFlash('ليس لديك صلاحية للوصول إلى هذه الصفحة', 'error');
            header('Location: index.php');
            exit;
        }
    }
    
    public function index() {
        $page = $_GET['page'] ?? 1;
        $perPage = 10;
        $search = $_GET['search'] ?? '';
        
        $units = $this->unitModel->getAll($page, $perPage, $search);
        $totalUnits = $this->unitModel->getTotalCount($search);
        $totalPages = ceil($totalUnits / $perPage);
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/units/index.php';
    }
    
    public function create() {
        if (!$this->auth->checkPermission('units', 'create')) {
            $this->session->setFlash('ليس لديك صلاحية لإضافة وحدات قياس', 'error');
            header('Location: units.php');
            exit;
        }
        
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'name' => trim($_POST['name']),
                    'code' => trim($_POST['code']),
                    'description' => trim($_POST['description'])
                ];
                
                // التحقق من صحة البيانات
                if (empty($data['name']) || empty($data['code'])) {
                    throw new Exception('اسم الوحدة ورمزها مطلوبان');
                }
                
                $unitId = $this->unitModel->create($data);
                
                $this->session->setFlash('تم إضافة الوحدة بنجاح', 'success');
                header('Location: units.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/units/create.php';
    }
    
    public function edit($id) {
        if (!$this->auth->checkPermission('units', 'update')) {
            $this->session->setFlash('ليس لديك صلاحية لتعديل وحدات القياس', 'error');
            header('Location: units.php');
            exit;
        }
        
        $unit = $this->unitModel->getById($id);
        if (!$unit) {
            $this->session->setFlash('الوحدة غير موجودة', 'error');
            header('Location: units.php');
            exit;
        }
        
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'name' => trim($_POST['name']),
                    'code' => trim($_POST['code']),
                    'description' => trim($_POST['description'])
                ];
                
                // التحقق من صحة البيانات
                if (empty($data['name']) || empty($data['code'])) {
                    throw new Exception('اسم الوحدة ورمزها مطلوبان');
                }
                
                $this->unitModel->update($id, $data);
                
                $this->session->setFlash('تم تحديث الوحدة بنجاح', 'success');
                header('Location: units.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/units/edit.php';
    }
    
    public function delete($id) {
        if (!$this->auth->checkPermission('units', 'delete')) {
            $this->session->setFlash('ليس لديك صلاحية لحذف وحدات القياس', 'error');
            header('Location: units.php');
            exit;
        }
        
        $unit = $this->unitModel->getById($id);
        if (!$unit) {
            $this->session->setFlash('الوحدة غير موجودة', 'error');
        } else {
            $this->unitModel->delete($id);
            $this->session->setFlash('تم حذف الوحدة بنجاح', 'success');
        }
        
        header('Location: units.php');
        exit;
    }
}
?>