<?php
// controllers/CurrencyController.php
require_once 'models/CurrencyModel.php';
require_once 'core/Auth.php';
require_once 'core/Session.php';

class CurrencyController {
    private $currencyModel;
    private $auth;
    private $session;
    
    public function __construct() {
        $this->currencyModel = new CurrencyModel();
        $this->auth = new Auth();
        $this->session = new Session();
        
        // التحقق من الصلاحيات
        if (!$this->auth->isLoggedIn()) {
            header('Location: login.php');
            exit;
        }
        
        if (!$this->auth->checkPermission('currencies', 'read')) {
            $this->session->setFlash('ليس لديك صلاحية للوصول إلى هذه الصفحة', 'error');
            header('Location: index.php');
            exit;
        }
    }
    
    public function index() {
        $page = $_GET['page'] ?? 1;
        $perPage = 10;
        $search = $_GET['search'] ?? '';
        
        $currencies = $this->currencyModel->getAll($page, $perPage, $search);
        $totalCurrencies = $this->currencyModel->getTotalCount($search);
        $totalPages = ceil($totalCurrencies / $perPage);
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/currencies/index.php';
    }
    
    public function create() {
        if (!$this->auth->checkPermission('currencies', 'create')) {
            $this->session->setFlash('ليس لديك صلاحية لإضافة عملات', 'error');
            header('Location: currencies.php');
            exit;
        }
        
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'name' => trim($_POST['name']),
                    'code' => trim($_POST['code']),
                    'symbol' => trim($_POST['symbol']),
                    'exchange_rate' => (float)$_POST['exchange_rate'],
                    'is_default' => isset($_POST['is_default']) ? 1 : 0
                ];
                
                // التحقق من صحة البيانات
                if (empty($data['name']) || empty($data['code'])) {
                    throw new Exception('اسم العملة ورمزها مطلوبان');
                }
                
                if ($data['exchange_rate'] <= 0) {
                    throw new Exception('سعر الصرف يجب أن يكون أكبر من صفر');
                }
                
                $currencyId = $this->currencyModel->create($data);
                
                $this->session->setFlash('تم إضافة العملة بنجاح', 'success');
                header('Location: currencies.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/currencies/create.php';
    }
    
    public function edit($id) {
        if (!$this->auth->checkPermission('currencies', 'update')) {
            $this->session->setFlash('ليس لديك صلاحية لتعديل العملات', 'error');
            header('Location: currencies.php');
            exit;
        }
        
        $currency = $this->currencyModel->getById($id);
        if (!$currency) {
            $this->session->setFlash('العملة غير موجودة', 'error');
            header('Location: currencies.php');
            exit;
        }
        
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'name' => trim($_POST['name']),
                    'code' => trim($_POST['code']),
                    'symbol' => trim($_POST['symbol']),
                    'exchange_rate' => (float)$_POST['exchange_rate'],
                    'is_default' => isset($_POST['is_default']) ? 1 : 0
                ];
                
                // التحقق من صحة البيانات
                if (empty($data['name']) || empty($data['code'])) {
                    throw new Exception('اسم العملة ورمزها مطلوبان');
                }
                
                if ($data['exchange_rate'] <= 0) {
                    throw new Exception('سعر الصرف يجب أن يكون أكبر من صفر');
                }
                
                $this->currencyModel->update($id, $data);
                
                $this->session->setFlash('تم تحديث العملة بنجاح', 'success');
                header('Location: currencies.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/currencies/edit.php';
    }
    
    public function delete($id) {
        if (!$this->auth->checkPermission('currencies', 'delete')) {
            $this->session->setFlash('ليس لديك صلاحية لحذف العملات', 'error');
            header('Location: currencies.php');
            exit;
        }
        
        try {
            $currency = $this->currencyModel->getById($id);
            if (!$currency) {
                $this->session->setFlash('العملة غير موجودة', 'error');
            } else {
                $this->currencyModel->delete($id);
                $this->session->setFlash('تم حذف العملة بنجاح', 'success');
            }
        } catch (Exception $e) {
            $this->session->setFlash($e->getMessage(), 'error');
        }
        
        header('Location: currencies.php');
        exit;
    }
}
?>