<?php
// models/UnitModel.php
require_once 'core/Database.php';

class UnitModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
   public function getAll($page = 1, $perPage = 10, $search = '') {
    $offset = ($page - 1) * $perPage;
    $where = [];
    $params = [];

    if (!empty($search)) {
        $where[] = "(name LIKE :search OR code LIKE :search2)";
        $params[':search'] = "%$search%";
        $params[':search2'] = "%$search%";
    }

    $where[] = "is_active = 1";
    $whereClause = "WHERE " . implode(" AND ", $where);

    $sql = "SELECT * FROM units
            {$whereClause}
            ORDER BY name ASC
            LIMIT :limit OFFSET :offset";

    $stmt = $this->db->getConnection()->prepare($sql);

    // ربط معاملات البحث
    foreach ($params as $key => $val) {
        $stmt->bindValue($key, $val);
    }

    $stmt->bindValue(':limit', (int)$perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);

    $stmt->execute();

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

    
    public function getById($id) {
        $sql = "SELECT * FROM units WHERE id = ? AND is_active = 1";
        $stmt = $this->db->query($sql, [$id]);
        return $stmt->fetch();
    }
    
    public function create($data) {
        $this->db->beginTransaction();
        
        try {
            $sql = "INSERT INTO units (name, code, description, created_at) 
                    VALUES (?, ?, ?, NOW())";
            
            $this->db->query($sql, [
                $data['name'],
                $data['code'],
                $data['description']
            ]);
            
            $unitId = $this->db->lastInsertId();
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'create', 'unit', $unitId, null, $data);
            
            $this->db->commit();
            
            return $unitId;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function update($id, $data) {
        $this->db->beginTransaction();
        
        try {
            $oldUnit = $this->getById($id);
            
            $sql = "UPDATE units SET name = ?, code = ?, description = ? 
                    WHERE id = ?";
            
            $this->db->query($sql, [
                $data['name'],
                $data['code'],
                $data['description'],
                $id
            ]);
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'update', 'unit', $id, $oldUnit, $data);
            
            $this->db->commit();
            
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function delete($id) {
        $unit = $this->getById($id);
        
        $sql = "UPDATE units SET is_active = 0 WHERE id = ?";
        $result = $this->db->query($sql, [$id]);
        
        // تسجيل النشاط
        if ($result) {
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'delete', 'unit', $id, $unit);
        }
        
        return $result;
    }
    
    public function getTotalCount($search = '') {
        $where = [];
        $params = [];
        
        if (!empty($search)) {
            $where[] = "(name LIKE ? OR code LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        $where[] = "is_active = 1";
        $whereClause = "WHERE " . implode(" AND ", $where);
        
        $sql = "SELECT COUNT(*) as total FROM units {$whereClause}";
        $stmt = $this->db->query($sql, $params);
        return $stmt->fetch()['total'];
    }
    
    public function getAllActive() {
        $sql = "SELECT id, name, code FROM units WHERE is_active = 1 ORDER BY name";
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }
    
    private function logActivity($userId, $action, $entityType, $entityId, $oldValues = null, $newValues = null) {
        try {
            $sql = "INSERT INTO activity_logs 
                    (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            
            $this->db->query($sql, [
                $userId,
                $action,
                $entityType,
                $entityId,
                $oldValues ? json_encode($oldValues, JSON_UNESCAPED_UNICODE) : null,
                $newValues ? json_encode($newValues, JSON_UNESCAPED_UNICODE) : null,
                $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ]);
        } catch (Exception $e) {
            error_log("خطأ في تسجيل النشاط: " . $e->getMessage());
        }
    }
}
?>