<?php
// models/TransactionModel.php
require_once 'core/Database.php';

class TransactionModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function recordInvoiceTransactions($invoiceId, $invoiceData, $items) {
        $this->db->beginTransaction();
        
        try {
            // 1. قيد العميل (مدين)
            $this->recordCustomerTransaction($invoiceId, $invoiceData);
            
            // 2. قيود الشركات الموردة (دائنة) للأصناف المرتبطة بشركات
            $this->recordCompanyTransactions($invoiceId, $items);
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    private function recordCustomerTransaction($invoiceId, $invoiceData) {
        // الحصول على الرصيد الحالي للعميل
        $currentBalance = $this->getCurrentBalance('customer', $invoiceData['customer_id']);
        $newBalance = $currentBalance + $invoiceData['total_amount_local'];
        
        $sql = "INSERT INTO transactions (date, account_type, account_id, related_invoice_id, 
                debit_amount, credit_amount, description, balance_after, created_at) 
                VALUES (CURDATE(), 'customer', ?, ?, ?, 0, ?, ?, NOW())";
        
        $this->db->query($sql, [
            $invoiceData['customer_id'],
            $invoiceId,
            $invoiceData['total_amount_local'],
            'فاتورة مبيعات - ' . $invoiceData['invoice_number'],
            $newBalance
        ]);
        
        // تحديث رصيد العميل
        $this->updateAccountBalance('customer', $invoiceData['customer_id'], $newBalance);
    }
    
    private function recordCompanyTransactions($invoiceId, $items) {
        $companyTotals = [];
        
        // تجميع المبالغ لكل شركة
        foreach ($items as $item) {
            if (!empty($item['company_id'])) {
                if (!isset($companyTotals[$item['company_id']])) {
                    $companyTotals[$item['company_id']] = 0;
                }
                $companyTotals[$item['company_id']] += $item['total_price'];
            }
        }
        
        // تسجيل الحركات للشركات
        foreach ($companyTotals as $companyId => $amount) {
            $currentBalance = $this->getCurrentBalance('company', $companyId);
            $newBalance = $currentBalance - $amount; // دائن (ناقص)
            
            $sql = "INSERT INTO transactions (date, account_type, account_id, related_invoice_id, 
                    debit_amount, credit_amount, description, balance_after, created_at) 
                    VALUES (CURDATE(), 'company', ?, ?, 0, ?, ?, ?, NOW())";
            
            $this->db->query($sql, [
                $companyId,
                $invoiceId,
                $amount,
                'مستحقات فاتورة',
                $newBalance
            ]);
            
            // تحديث رصيد الشركة
            $this->updateAccountBalance('company', $companyId, $newBalance);
        }
    }
    
    public function recordVoucherTransaction($voucherData) {
        $this->db->beginTransaction();
        
        try {
            $currentBalance = $this->getCurrentBalance($voucherData['related_to'], $voucherData['related_id']);
            
            if ($voucherData['type'] == 'receipt') {
                // سند قبض: زيادة الصندوق/البنك، تخفيض رصيد العميل/الشركة
                $newBalance = $currentBalance - $voucherData['amount_local'];
                $debit = 0;
                $credit = $voucherData['amount_local'];
            } else {
                // سند صرف: تخفيض الصندوق/البنك، زيادة رصيد العميل/الشركة
                $newBalance = $currentBalance + $voucherData['amount_local'];
                $debit = $voucherData['amount_local'];
                $credit = 0;
            }
            
            $sql = "INSERT INTO transactions (date, account_type, account_id, related_voucher_id, 
                    debit_amount, credit_amount, description, balance_after, created_at) 
                    VALUES (CURDATE(), ?, ?, ?, ?, ?, ?, ?, NOW())";
            
            $this->db->query($sql, [
                $voucherData['related_to'],
                $voucherData['related_id'],
                $voucherData['id'],
                $debit,
                $credit,
                $voucherData['description'],
                $newBalance
            ]);
            
            // تحديث الرصيد
            $this->updateAccountBalance($voucherData['related_to'], $voucherData['related_id'], $newBalance);
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    private function getCurrentBalance($accountType, $accountId) {
        $sql = "SELECT current_balance FROM ";
        
        switch ($accountType) {
            case 'customer':
                $sql .= "customers";
                break;
            case 'company':
                $sql .= "companies";
                break;
            case 'cash':
            case 'bank':
                // للصناديق والبنوك، نستخدم جدول مختلف أو نعيد 0
                return 0;
        }
        
        $sql .= " WHERE id = ?";
        $stmt = $this->db->query($sql, [$accountId]);
        $result = $stmt->fetch();
        
        return $result ? $result['current_balance'] : 0;
    }
    
    private function updateAccountBalance($accountType, $accountId, $newBalance) {
        $sql = "UPDATE ";
        
        switch ($accountType) {
            case 'customer':
                $sql .= "customers";
                break;
            case 'company':
                $sql .= "companies";
                break;
            case 'cash':
            case 'bank':
                // للصناديق والبنوك، نستخدم جدول مختلف
                return;
        }
        
        $sql .= " SET current_balance = ? WHERE id = ?";
        $this->db->query($sql, [$newBalance, $accountId]);
    }
    
    public function getAccountStatement($accountType, $accountId, $startDate = null, $endDate = null) {
        $where = ["account_type = ?", "account_id = ?"];
        $params = [$accountType, $accountId];
        
        if ($startDate) {
            $where[] = "DATE(date) >= ?";
            $params[] = $startDate;
        }
        
        if ($endDate) {
            $where[] = "DATE(date) <= ?";
            $params[] = $endDate;
        }
        
        $whereClause = "WHERE " . implode(" AND ", $where);
        
        $sql = "SELECT * FROM transactions 
                {$whereClause}
                ORDER BY date, id";
        
        $stmt = $this->db->query($sql, $params);
        return $stmt->fetchAll();
    }
}
?>