<?php
// models/CurrencyModel.php
require_once 'core/Database.php';

class CurrencyModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
   public function getAll($page = 1, $perPage = 10, $search = '') {
    $offset = ($page - 1) * $perPage;
    $where = [];
    $params = [];

    if (!empty($search)) {
        $where[] = "(name LIKE :search OR code LIKE :search2)";
        $params[':search'] = "%$search%";
        $params[':search2'] = "%$search%";
    }

    $where[] = "is_active = 1";
    $whereClause = "WHERE " . implode(" AND ", $where);

    $sql = "SELECT * FROM currencies
            {$whereClause}
            ORDER BY is_default DESC, name ASC
            LIMIT :limit OFFSET :offset";

    $stmt = $this->db->getConnection()->prepare($sql);

    // ربط معاملات البحث
    foreach ($params as $key => $val) {
        $stmt->bindValue($key, $val);
    }

    // ربط LIMIT و OFFSET كأعداد صحيحة
    $stmt->bindValue(':limit', (int)$perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);

    $stmt->execute();

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

    
    public function getById($id) {
        $sql = "SELECT * FROM currencies WHERE id = ? AND is_active = 1";
        $stmt = $this->db->query($sql, [$id]);
        return $stmt->fetch();
    }
    
    public function getDefaultCurrency() {
        $sql = "SELECT * FROM currencies WHERE is_default = 1 AND is_active = 1 LIMIT 1";
        $stmt = $this->db->query($sql);
        return $stmt->fetch();
    }
    
    public function create($data) {
        $this->db->beginTransaction();
        
        try {
            // إذا كانت العملة افتراضية، إلغاء التحديد من العملات الأخرى
            if (isset($data['is_default']) && $data['is_default']) {
                $this->unsetDefaultCurrencies();
            }
            
            $sql = "INSERT INTO currencies (name, code, symbol, exchange_rate, is_default, created_at) 
                    VALUES (?, ?, ?, ?, ?, NOW())";
            
            $this->db->query($sql, [
                $data['name'],
                $data['code'],
                $data['symbol'],
                $data['exchange_rate'],
                $data['is_default'] ?? 0
            ]);
            
            $currencyId = $this->db->lastInsertId();
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'create', 'currency', $currencyId, null, $data);
            
            $this->db->commit();
            
            return $currencyId;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function update($id, $data) {
        $this->db->beginTransaction();
        
        try {
            $oldCurrency = $this->getById($id);
            
            // إذا كانت العملة افتراضية، إلغاء التحديد من العملات الأخرى
            if (isset($data['is_default']) && $data['is_default']) {
                $this->unsetDefaultCurrencies();
            }
            
            $sql = "UPDATE currencies SET name = ?, code = ?, symbol = ?, exchange_rate = ?, 
                    is_default = ?, updated_at = NOW() 
                    WHERE id = ?";
            
            $this->db->query($sql, [
                $data['name'],
                $data['code'],
                $data['symbol'],
                $data['exchange_rate'],
                $data['is_default'] ?? 0,
                $id
            ]);
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'update', 'currency', $id, $oldCurrency, $data);
            
            $this->db->commit();
            
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function delete($id) {
        $currency = $this->getById($id);
        
        // منع حذف العملة الافتراضية
        if ($currency && $currency['is_default']) {
            throw new Exception('لا يمكن حذف العملة الافتراضية');
        }
        
        $sql = "UPDATE currencies SET is_active = 0, updated_at = NOW() WHERE id = ?";
        $result = $this->db->query($sql, [$id]);
        
        // تسجيل النشاط
        if ($result) {
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'delete', 'currency', $id, $currency);
        }
        
        return $result;
    }
    
    public function getTotalCount($search = '') {
        $where = [];
        $params = [];
        
        if (!empty($search)) {
            $where[] = "(name LIKE ? OR code LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        $where[] = "is_active = 1";
        $whereClause = "WHERE " . implode(" AND ", $where);
        
        $sql = "SELECT COUNT(*) as total FROM currencies {$whereClause}";
        $stmt = $this->db->query($sql, $params);
        return $stmt->fetch()['total'];
    }
    
    public function getAllActive() {
        $sql = "SELECT id, name, code, symbol, exchange_rate FROM currencies WHERE is_active = 1 ORDER BY name";
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }
    
    private function unsetDefaultCurrencies() {
        $sql = "UPDATE currencies SET is_default = 0, updated_at = NOW() WHERE is_default = 1";
        $this->db->query($sql);
    }
    
    private function logActivity($userId, $action, $entityType, $entityId, $oldValues = null, $newValues = null) {
        try {
            $sql = "INSERT INTO activity_logs 
                    (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            
            $this->db->query($sql, [
                $userId,
                $action,
                $entityType,
                $entityId,
                $oldValues ? json_encode($oldValues, JSON_UNESCAPED_UNICODE) : null,
                $newValues ? json_encode($newValues, JSON_UNESCAPED_UNICODE) : null,
                $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ]);
        } catch (Exception $e) {
            error_log("خطأ في تسجيل النشاط: " . $e->getMessage());
        }
    }
}
?>